# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import time

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_engine_height
from psdk.platform.util.product_util import get_ctrl_id_by_node_id
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version

# 定义告警导出日志的事件ID
ALM_EXPORT_FILE_EVENT_ID = "0x200F00310005"
# 定义导出日志可能出现问题的次数最大值
ALM_MAX_EXPORT_COUNT = 30
Mon = {"Jan": "01", "Feb": "02", "Mar": "03", "Apr": "04", "May": "05", "Jun": "06",
       "Jul": "07", "Aug": "08", "Sep": "09", "Oct": "10", "Nov": "11", "Dec": "12", }
TIME_MON_POS = 2
TIME_DAY_POS = 3
TIME_TIME_POS = 4
TIME_YEAR_POS = 5
TIME_PROCESS_NAME_POS = 6

pass_version_dic = {
    "V500R007C60SPC300": "SPH330"
}


class CheckItem(BaseCheckItem):

    @staticmethod
    def _has_ip_enclosure(enclosures):
        """
        是否含有smart的框代表智能硬盘框
        """
        return bool([enclosure for enclosure in enclosures if "smart" in enclosure.get("Type")])

    def execute(self):
        # 优先检查补丁版本，SPH330解决了此问题
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        base_version = version_info.get("base_version").get("Current Version")

        if base_version in pass_version_dic and compare_patch_version(patch_version,
            pass_version_dic.get(base_version)) >= 0:
            return CheckStatus.PASS, ""
        # 检查方法步骤1：使用dsl执行命令查询框的信息
        enclosures = self.dsl("exec_developer 'show enclosure' | horizontal_parser")
        if not self._has_ip_enclosure(enclosures):
            return CheckStatus.PASS, ""

        # 检查方法步骤2：使用dsl执行命令查询进程的启动时间
        result_info = self.dsl("exec_on_all {}", self._get_result_info)
        abnormal_ctrls = []
        engine_height = get_engine_height(self.dsl, self.context)
        for node_id, result in result_info.items():
            if result:
                ctrl_id = get_ctrl_id_by_node_id(node_id, engine_height)
                abnormal_ctrls.append((ctrl_id, result))
        err_msg = []
        for abnormal_ctrl in abnormal_ctrls:
            err_msg.append(self.get_msg("ctrl.id.not.pass", abnormal_ctrl[0], abnormal_ctrl[1]))
        if err_msg:
            return CheckStatus.WARNING, "\n".join(err_msg)
        return CheckStatus.PASS, ""

    def _export_file_count_is_over_limit(self, event_info):
        """
        根据启动时间获取指定告警ID出现的次数是否大于最大场景30次
        """
        count = 0
        for item in event_info:
            if ALM_EXPORT_FILE_EVENT_ID == item.get("ID"):
                count = count + 1
        return count > ALM_MAX_EXPORT_COUNT

    def _get_process_start_time(self, process_info):
        """
        获取进程的启动时间
        3538373 Wed May 25 15:22:51 2022 /OSM/bin/alarm 转化为 2022-05-25/15:22:51
        """
        start_time = ""
        # 捕获/OSM/bin/alarm信息
        for item in process_info:
            if "/OSM/bin/alarm" in item:
                item_list = item.split()
                if len(item_list) < TIME_PROCESS_NAME_POS:
                    continue
                year = item_list[TIME_YEAR_POS]
                month = Mon.get(item_list[TIME_MON_POS], "01")
                day = item_list[TIME_DAY_POS]
                _time = item_list[TIME_TIME_POS]
                start_time = "{}-{}-{}/{}".format(year, month, day, _time)
                break
        return start_time

    def _get_result_info(self):
        """
        获取单个节点是否存在超过30次的event事件ID.
        """
        process_info = self.dsl("exec_mini_system 'ps -eo pid,lstart,cmd' | splitlines ")
        process_time = self._get_process_start_time(process_info)
        if not process_time:
            # 获取信息失败 重试一次
            time.sleep(5)
            process_info = self.dsl("exec_mini_system 'ps -eo pid,lstart,cmd' | splitlines ")
            process_time = self._get_process_start_time(process_info)
            if not process_time:
                return self.get_msg("get.msg.not.pass")

        # 检查方法步骤3：使用dsl执行命令查询某个事件ID出现的次数
        event_info = self.dsl(
            "exec_cli 'show event from_time={} object_type=49 level=informational' | horizontal_parser".format(
                process_time))
        if self._export_file_count_is_over_limit(event_info):
            return self.get_msg("check.not.pass")
        return ""
