# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2022. All rights reserved.
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version
from psdk.dsl.dsl_common import get_engine_height
from psdk.platform.util.product_util import get_ctrl_id_by_node_id

PMS_LUN_CHECK_STATUS_UNKNOWN = 0
PMS_LUN_CHECK_STATUS_PASS = 1
PMS_LUN_CHECK_STATUS_NOT_PASS = -1
PMS_LUN_CHECK_STATUS_NO_PROCESS = -2
# 检测阈值
PMS_LUN_WARN_THRESHOLD = 0.85


class CheckItem(BaseCheckItem):
    def execute(self):
        if not self.check_pms_service_status():
            self.logger.info("Retain historical monitoring data in Monitoring Settings is off. check not support")
            return CheckStatus.NOT_SUPPORT, ""
        # 判断当前版本是否为6.0.1以及补丁版本是否大于SPH21或者是否为6.1.0以及补丁版本是否大于SPH9直接默认通过
        if self.check_version_pass():
            return CheckStatus.PASS, ""
        # 版本判断之后再进行pmslun判断
        usage_results = self.dsl("exec_on_all {}", self.check_lun_usage_status)
        engine_height = get_engine_height(self.dsl, self.context)
        # PMS只会在其中某个控制器运行，所以只要有PMS进程存在时的判断结果，就可直接返回
        for node_id, check_status in usage_results.items():
            if check_status == PMS_LUN_CHECK_STATUS_PASS:
                return CheckStatus.PASS, ""
            elif check_status == PMS_LUN_CHECK_STATUS_NOT_PASS:
                return CheckStatus.NOT_PASS, self.get_msg("check.not.pass")
            elif check_status == PMS_LUN_CHECK_STATUS_UNKNOWN:
                ctrl_id = get_ctrl_id_by_node_id(node_id, engine_height)
                return CheckStatus.NOT_CHECK, self.get_msg("check.unknown", ctrl_id)
        # 所有控制器未检测到有效进程，则检测不通过
        return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.no.pid")

    def check_lun_usage_status(self):
        java_pid_results = self.dsl("exec_mini_system 'ps -C java' | splitlines ")
        if not self.check_pid_valid(java_pid_results):
            return PMS_LUN_CHECK_STATUS_NO_PROCESS

        lun_usage_results = self.dsl("exec_mini_system '{}' | splitlines",
                                     "maintain -c grep -p 'Disk[[:space:]]UsedRate[[:space:]]is'"
                                     " /home/permitdir/coffer_log/log/pms/logs/systemReporter-info.log")
        pms_lun_usage_status = self.check_lun_usage(lun_usage_results)
        if pms_lun_usage_status == PMS_LUN_CHECK_STATUS_UNKNOWN and "No space left on device" in \
                self.dsl("exec_mini_system '{}'",
                         "maintain -c grep -p "
                         "'No[[:space:]]space[[:space:]]left[[:space:]]on[[:space:]]device' "
                         "/home/permitdir/coffer_log/log/pms/logs/systemReporter-info.log"):
            return PMS_LUN_CHECK_STATUS_NOT_PASS
        return pms_lun_usage_status

    def check_version_pass(self):
        version_info = get_version_info(self.dsl)
        if not version_info:
            self.logger.info("no version info. check pass")
            return True
        # 获取产品版本和补丁版本,获取不到产品版本和补丁版本返回通过
        pro_version = version_info.get("base_version").get("Current Version")
        sph_version = version_info.get("patch_version").get("Current Version")
        if not pro_version or not sph_version:
            self.logger.info("not contain base or patch version. check pass")
            return True
        # 判断当前版本是否为6.0.1以及补丁版本是否大于SPH21或者是否为6.1.0以及补丁版本是否大于SPH9直接默认通过
        if ("6.0.1" in pro_version and compare_patch_version(sph_version, 'SPH21') >= 0) or \
                ("6.1.0" in pro_version and compare_patch_version(sph_version, 'SPH9') >= 0):
            self.logger.info("base_version {},patch_version {}. pmsLun check pass".format(pro_version, sph_version))
            return True
        return False

    def check_pms_service_status(self):
        status_results = self.dsl("exec_cli 'show performance retention_strategy' | splitlines")
        retention_strategy = ""
        for status_result in status_results:
            if "Retention Switch" in status_result:
                retention_strategy = status_result.split(":")[1].strip()
                break
        return "on" == retention_strategy

    @staticmethod
    def check_pid_valid(java_pid_results):
        return bool([java_pid_result for java_pid_result in java_pid_results
                     if ("ps -C java" not in java_pid_result and "java" in java_pid_result)])

    @staticmethod
    def check_lun_usage(lun_usage_results):
        usage_num = -1.0
        lun_usage_results = lun_usage_results[::-1]
        for lun_usage_result in lun_usage_results:
            if re.match("^.*Disk UsedRate is\s*(\d+(\.\d+)?)$", lun_usage_result):
                usage_num = float(lun_usage_result.split(' ')[-1])
                break
        if usage_num == -1.0:
            return PMS_LUN_CHECK_STATUS_UNKNOWN
        if usage_num < PMS_LUN_WARN_THRESHOLD:
            return PMS_LUN_CHECK_STATUS_PASS
        return PMS_LUN_CHECK_STATUS_NOT_PASS
