# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.protocol.cli_common import get_system_version_without_ret
from psdk.platform.util.product_util import compare_patch_version
from psdk.dsl import fault_mode as ft
from psdk.checkitem.common.fru_common import is_selected_fan_in_disk_enclosure
from psdk.dsl.dsl_common import is_single_device


risk_version_dict = {
    "V300R006C50SPC100": "V300R006C50SPH131",
    "V300R006C20": "V300R006C20SPH038",
    "V500R007C10": "V500R007C10SPH038",
    "V500R007C30SPC100": "V500R007C30SPH131",
    "V500R007C60SPC300": "V500R007C60SPH322",
    "V500R007C61": "V500R007C61SPH030",
}


class CheckItem(BaseCheckItem):
    def execute(self):
        # 判断当前更换是否为硬盘框风扇
        if is_selected_fan_in_disk_enclosure(self):
            self.logger.info("fan in disk enclosure, check pass")
            return CheckStatus.PASS, ""
        # 单控直接通过
        if is_single_device(self.dsl):
            return CheckStatus.PASS, ""
        # 获取版本信息
        flag, software_version, hot_patch_version = \
            get_system_version_without_ret(self.dsl("exec_cli 'show upgrade package'"))
        if is_no_risk_version(software_version, hot_patch_version):
            return CheckStatus.PASS, ""
        # 判断是否存在文件系统远程复制
        remote_devices = self.dsl("exec_cli 'show remote_replication unified list_type=fs|"
                                  "filterColumn include columnList=ID,Remote\sDevice\sID' |"
                                  " horizontal_parser", return_if={ft.FindException: []})
        if not remote_devices:
            return CheckStatus.PASS, ""
        # 判断符合条件id
        device_id_list = list(set([remote_device.get("Remote Device ID") for remote_device in remote_devices
                              if 1 <= int(remote_device.get("Remote Device ID")) <= 63]))
        if not device_id_list:
            return CheckStatus.PASS, ""
        # 根据版本判断建议
        if software_version.startswith("V3") and software_version <= "V300R006C50SPC100" or \
                software_version.startswith("V5") and software_version <= "V500R007C61":
            suggestion = self.set_error_msg(software_version)
        elif "Kunpeng" in self.context.dev_node.version:
            suggestion = self.get_msg("check.suggestion.kunpeng")
        else:
            suggestion = self.get_msg("check.suggestion.other")
        return CheckStatus.NOT_PASS, self.get_msg("check.error.msg", ", ".join(device_id_list)) + \
            self.get_msg("check.suggestion.keyword") + suggestion

    def set_error_msg(self, software_version):
        suggestion_p_version = "V300R006C50SPC100" if is_v3(software_version) else "V500R007C30SPC100"
        if software_version in risk_version_dict:
            suggestion = self.get_msg("check.suggestion.upgrade.patch", risk_version_dict.get(software_version))
        else:
            suggestion = self.get_msg("check.suggestion.upgrade.software",
                                      suggestion_p_version, risk_version_dict.get(suggestion_p_version))
        return suggestion


def is_no_risk_version(software_version, hot_patch_version):
    return software_version in risk_version_dict and compare_patch_version(
            hot_patch_version, risk_version_dict.get(software_version)) >= 0


def is_v3(software_version):
    return software_version.startswith("V3")

