# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import sys
import os
import re
import ast
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_version, compare_patch_version
from psdk.platform.base.logger import log

CHECK_613_FLAG_PATCH_START_VERSION = "SPH26"
CHECK_BASE_VERSION = "6.1.3"
CHECK_615_FLAG_PATCH_START_VERSION = "SPH18"
CHECK_BASE_VERSION_615 = "6.1.5"



class CheckItem(BaseCheckItem):
    def execute(self):
        version_info = get_version_info(self.dsl)
        base_version = version_info.get("base_version").get("Current Version")
        patch_version = version_info.get("patch_version").get("Current Version")
        if not self.is_version_match(base_version, patch_version):
            return CheckStatus.PASS, ""
        if (not self.is_2p()) or (not self.is_h8_share_card()) or (not self.is_h8_card_fault()): 
            return CheckStatus.PASS, ""
        return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion", 
            (self.join_version(base_version, patch_version), ), self.recommended_version(base_version))


    def is_2p(self):
        #检查是否是高端2p单板
        cpu_info = self.dsl("exec_diagnose 'mpdk showcpuult'")
        cpu_num = re.search(r'(?<=Total vnode num: )\d+', cpu_info)
        if cpu_num is None or int(cpu_num.group()) != 2:
            return False
        return True


    def is_h8_share_card(self):
        #检查是否是H8槽位的共享卡
        selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
        self.logger.info(selected_fru.get("healthStatus"))
        if ("583" in selected_fru.get("intModel")) and ("IOM.H8" in selected_fru.get("location")):
            return True
        return False


    def is_version_match(self, base_version, patch_version):
        if ((compare_version(base_version, CHECK_BASE_VERSION) == 0) and 
            (compare_patch_version(patch_version, CHECK_613_FLAG_PATCH_START_VERSION) < 0)):
            return True
        if ((compare_version(base_version, CHECK_BASE_VERSION_615) == 0) and 
             (compare_patch_version(patch_version, CHECK_615_FLAG_PATCH_START_VERSION) < 0)):
            return True
        return False

    
    def is_h8_card_fault(self):
        #检查H8槽位的卡状态是否为fault
        card_info = self.dsl("exec_cli 'show interface_module'")
        infos = card_info.split('\n')
        for line in infos:
            if "IOM.H8" in line and "Fault" in line:
                self.logger.info(line)
                return True
        return False


    def join_version(self, base_version, patch_version):
        if ("--" in patch_version):
            return base_version
        return base_version + ' ' + patch_version

    
    def recommended_version(self, base_version):
        if base_version == "6.1.3":
            return ("6.1.3 SPH26",)
        else:
            return ("6.1.5 SPH18",)
