# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import ast
import json
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus


class CheckItem(BaseCheckItem):
    def execute(self):
        self.logger.info("Checking the box after replace.")
        try:
            selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
            box_cur = selected_fru.get("name")
        except Exception as e:
            self.logger.error("Get box Id from selected_fru failed.")
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.boxId", "item.suggestion.boxId")

        box_info = self.dsl("exec_cli 'show box general' | horizontal_parser")
        box_status_dict = {}
        for box in box_info:
            box_id = box.get("ID")
            box_status = box.get("Health Status")
            box_model = box.get("Model")
            box_status_dict[box_id] = box_status
            if box_cur == box_id and "disk" in box_model:
                # 硬盘盒直接检查通过
                self.logger.info("The type of box {} is disk, return PASS.".format(box_cur))
                return CheckStatus.PASS, ""
        if box_cur not in box_status_dict.keys():
            self.logger.error("The box({}) is not found on device.".format(box_cur))
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.boxId", "item.suggestion.boxId")
        if box_status_dict.get(box_cur) != "Normal":
            self.logger.error("The box({}) status({}) is not normal.".format(box_cur, box_status_dict.get(box_cur)))
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.boxId", "item.suggestion.boxId")

        return self.check_box_intfcard_model(box_cur)

    def check_box_intfcard_model(self, box_cur):
        intfcard_dict_pre = json.loads(self.context.execute_env.ori_env.get("preCheckBoxIntfCard"))
        self.logger.info("The intfcard_dict_pre get from context is: {}".format(intfcard_dict_pre))
        if not intfcard_dict_pre:
            self.logger.error("The pre-check box intfcard info get from context is empty, return PASS.")
            return CheckStatus.PASS, ""

        intfcard_dict_cur = {}
        module_info = self.dsl("exec_rest '/intf_module'")
        for intf_info in module_info:
            location = intf_info.get("LOCATION")
            model_num = intf_info.get("MODEL")
            running_status = str(intf_info.get("RUNNINGSTATUS"))
            # 只检查正常状态的接口卡，"RUNNINGSTATUS": "2"
            if running_status != "2":
                continue
            if box_cur in location:
                intfcard_dict_cur[location] = model_num

        self.logger.info("The intfcard_dict_cur is: {}".format(intfcard_dict_cur))

        if len(intfcard_dict_cur.keys()) < len(intfcard_dict_pre.keys()):
            self.logger.error("The box intfcard number({}) is different from pre-check({})."
                              .format(len(intfcard_dict_cur.keys()), len(intfcard_dict_pre.keys())))
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.number", "item.suggestion.number")

        for intf_id, model in intfcard_dict_pre.items():
            if intf_id not in intfcard_dict_cur.keys():
                self.logger.error("The pre-check intfcard({}) is not found in current box.".format(intf_id))
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
            if intfcard_dict_cur.get(intf_id) != model:
                self.logger.error("The pre-check intfcard({}) model({}) is different from the current model({})."
                                  .format(intf_id, model, intfcard_dict_cur.get(intf_id)))
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")

        self.logger.info("Checking the box after replace PASS.")
        return CheckStatus.PASS, ""
