# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import ast
import json
import re
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus


class CheckItem(BaseCheckItem):
    write_data = {}
    # 存放漂移组id
    flv_id_list = []
    # 根据漂移组id，保存成员端口，格式：flv_member_dict[flv_id] = {p1, p2, ...}
    flv_member_dict = {}

    # 避免脚本重复调用时全局参数被重复设置或追加内容，统一进行初始化处理
    def init_global_param(self):
        self.write_data = {}
        self.flv_id_list = []
        self.flv_member_dict = {}

    def execute(self):
        self.logger.info("Checking the redundancy of box before replace.")

        self.init_global_param()

        try:
            selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
            box_cur = selected_fru.get("name")
        except Exception as e:
            self.logger.error("Get box Id from selected_fru failed.")
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.boxId", "item.suggestion.boxId")

        # 获取对称平面的盒Id
        box_check = ""
        if "L" in box_cur:
            box_check = box_cur.replace("L", "R")
        elif "R" in box_cur:
            box_check = box_cur.replace("R", "L")

        self.logger.info("The selected and to-be-checked boxs are {} and {}".format(box_cur, box_check))

        box_info = self.dsl("exec_cli 'show box general' | horizontal_parser")
        box_status_dict = {}
        for box in box_info:
            box_id = box.get("ID")
            box_status = box.get("Health Status")
            box_model = box.get("Model")
            box_status_dict[box_id] = box_status
            if box_cur == box_id and "disk" in box_model:
                # 硬盘盒直接检查通过
                self.logger.info("The type of box {} is disk, return PASS.".format(box_cur))
                return CheckStatus.PASS, ""
        if box_cur not in box_status_dict.keys() or box_check not in box_status_dict.keys():
            self.logger.error("The box {} or {} is not found on device.".format(box_cur, box_check))
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.boxId", "item.suggestion.boxId")
        # 对称平面的接口卡盒不是normal的
        if box_status_dict.get(box_check) != "Normal":
            self.logger.error("The check box({}) status({}) is not normal."
                              .format(box_check, box_status_dict.get(box_check)))
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.boxId", "item.suggestion.boxId")

        # 收集漂移组告警事件信息，保存到全局变量里
        self.query_failover_group_events()

        return self.check_box_intfcard_model(box_cur, box_check)

    def check_box_intfcard_model(self, box_cur, box_check):
        intfcard_dict_cur = {}
        intfcard_dict_check = {}
        module_info = self.dsl("exec_rest '/intf_module'")
        for intf_info in module_info:
            location = intf_info.get("LOCATION")
            model_num = intf_info.get("MODEL")
            running_status = str(intf_info.get("RUNNINGSTATUS"))
            # 只检查正常状态的接口卡，"RUNNINGSTATUS": "2"
            if running_status != "2":
                continue
            if box_cur in location:
                intfcard_dict_cur[location] = model_num
            elif box_check in location:
                intfcard_dict_check[location] = model_num

        self.write_data["preCheckBoxIntfCard"] = json.dumps(intfcard_dict_cur)

        if len(intfcard_dict_cur.keys()) == 0:
            self.logger.info("There is no running intfcard in the selected box({}), return PASS.".format(box_cur))
            self.write_intfcard_data_to_cache()
            return CheckStatus.PASS, ""

        ret, err_msg = self.box_info_check_func(intfcard_dict_cur, intfcard_dict_check)
        if ret != CheckStatus.PASS:
            self.logger.error("The box intfcard model check failed.")
            return ret, err_msg
        self.logger.info("The box intfcard model check PASS.")

        ret, err_msg = self.check_box_intfcard_port(box_cur, box_check)
        if ret != CheckStatus.PASS:
            self.logger.error("The box intfcard port check failed.")
            return ret, err_msg

        self.write_intfcard_data_to_cache()
        return CheckStatus.PASS, ""

    # 通过端口逻辑类型，检查当前接口卡盒内link-up的端口在对称平面的接口卡盒内是否有同类型、link-up的端口
    def check_box_intfcard_port(self, box_cur, box_check):
        # 由于端口的物理类型有多种（roce、eth），因此将端口逻辑类型按照物理类型分类保存
        # 后续对两个接口卡盒内相同物理类型的端口进行检查
        # 如：port_dict = {"roce":{"port_id_1":"0"}, "eth":{"port_id_2":"3"}}
        port_cur_dict = {}
        port_check_dict = {}
        eth_port_info = self.dsl("exec_rest '/eth_port'")
        if not eth_port_info:
            return CheckStatus.PASS, ""
        for eth_port_item in eth_port_info:
            port_id = eth_port_item.get("LOCATION")
            parent_type = str(eth_port_item.get("PARENTTYPE"))
            running_status = str(eth_port_item.get("RUNNINGSTATUS"))
            logic_type = str(eth_port_item.get("LOGICTYPE"))
            physical_type = str(eth_port_item.get("physicalType"))
            # 过滤不是接口卡类型，以及不是Link up状态的端口
            if parent_type != "209" or running_status != "10":
                continue

            if physical_type not in port_cur_dict.keys():
                port_cur_dict[physical_type] = {}
            if physical_type not in port_check_dict.keys():
                port_check_dict[physical_type] = {}

            if box_cur in port_id:
                self.logger.info("Add port({}) physical_type({}) logic_type({}) to box_cur({})."
                                 .format(port_id, physical_type, logic_type, box_cur))
                port_cur_dict[physical_type][port_id] = logic_type
            elif box_check in port_id:
                self.logger.info("Add port({}) physical_type({}) logic_type({}) to box_check({})."
                                 .format(port_id, physical_type, logic_type, box_check))
                port_check_dict[physical_type][port_id] = logic_type

        self.logger.info("The port_cur_dict is: {}".format(port_cur_dict))
        self.logger.info("The port_check_dict is: {}".format(port_check_dict))

        if len(port_cur_dict.keys()) == 0 and len(port_check_dict.keys()) == 0:
            self.logger.info("There is no link_up port in the selected box({}), return PASS.".format(box_cur))
            return CheckStatus.PASS, ""

        for physical_type in port_cur_dict.keys():
            if physical_type not in port_check_dict.keys():
                self.logger.error("The physicalType({}) port is not found int the checked box.".format(physical_type))
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.boxId", "item.suggestion.boxId")

            ret, err_msg = self.box_info_check_func(port_cur_dict[physical_type], port_check_dict[physical_type])
            if ret != CheckStatus.PASS:
                self.logger.error("The box intfcard link_up port check failed.")
                return ret, err_msg

            # link-up的端口冗余检查完成后，检查端口是否有对应的漂移组告警
            ret, err_msg = self.check_failover_group_event_with_port(port_cur_dict[physical_type].keys())
            if ret != CheckStatus.PASS:
                self.logger.error("There is failover group event for port, check failed.")
                return ret, err_msg

        self.logger.info("The box intfcard link_up port check PASS.")
        return CheckStatus.PASS, ""

    # 检查端口是否在漂移组成员端口不冗余告警对应的漂移组里
    def check_failover_group_event_with_port(self, cur_port_list):
        if not self.flv_id_list:
            return CheckStatus.PASS, ""

        for flv_id in self.flv_id_list:
            duplicate_ports = set(cur_port_list) & set(self.flv_member_dict[flv_id])
            if duplicate_ports:
                self.logger.error("There is failover group event with ports {}".format(duplicate_ports))
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.flv", "item.suggestion.flv")
        return CheckStatus.PASS, ""

    def box_info_check_func(self, current_info, check_info):
        for location_cur, value_cur in current_info.items():
            if value_cur not in check_info.values():
                self.logger.error("The value({}) of location_cur({}) is not found in the checked box."
                                  .format(value_cur, location_cur))
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
        return CheckStatus.PASS, ""

    def write_intfcard_data_to_cache(self):
        self.logger.info("The write_data is {}".format(self.write_data))
        for cache_key, cache_value in self.write_data.items():
            try:
                self.context.execute_env.ori_env.put(cache_key, cache_value)
            except Exception:
                self.logger.info("Set write_data to context env failed.")

    def get_unrecovered_failover_group_events(self, event_id):
        # 执行命令获取未恢复的漂移组端口成员不冗余告警事件
        cli_cmd = "exec_cli 'show event event_id={} status=unrecovered' | horizontal_parser".format(event_id)
        return self.dsl(cli_cmd)

    def get_event_details(self, sequence):
        # 执行命令获取事件的详细信息
        return self.dsl("exec_cli 'show event sequence={}' | vertical_parser".format(sequence))

    def get_failover_group_members(self, flv_id, member_type):
        # 执行命令获取漂移组端口成员信息
        cli_cmd = ("exec_cli 'show failover_group member failover_group_id={} type={}' | "
                   "horizontal_parser").format(flv_id, member_type)
        return self.dsl(cli_cmd)

    # 收集设备上漂移组端口不冗余告警事件对应的漂移组信息
    def query_failover_group_events(self):
        # 中、高端环境漂移组告警ID不一样，中端环境：0xF01210001，高端环境：0xF01210002，A800是高端环境
        flv_events = self.get_unrecovered_failover_group_events("0xF01210002")
        if not flv_events:
            return

        # 使用正则表达式，从事件详情的字符串中取出漂移组的id，字符串内容示例：xxx (ID 0, name System-defined) xxx
        pattern = r"ID\s+(\d+)"
        for event in flv_events:
            sequence = event.get("Sequence")
            flv_event_details = self.get_event_details(sequence)
            detail_str = flv_event_details[0].get("Detail")
            matches = re.search(pattern, detail_str)
            if not matches:
                continue
            flv_id = matches.group(1)
            self.flv_id_list.append(flv_id)

        self.query_failover_group_members()

    # 查询漂移组的成员端口列表
    def query_failover_group_members(self):
        if not self.flv_id_list:
            return

        # 漂移组端口成员类型
        flv_member_types = ("eth_port", "bond_port", "vlan")

        # 根据漂移组id查询漂移组不同类型的成员
        for flv_id in self.flv_id_list:
            if not self.flv_member_dict.get(flv_id):
                self.flv_member_dict[flv_id] = set()

            for member_type in flv_member_types:
                flv_members = self.get_failover_group_members(flv_id, member_type)
                if not flv_members:
                    continue
                port_list = self.get_ports_from_flv_members(member_type, flv_members)
                self.flv_member_dict[flv_id].update(list(port_list))

    # 漂移组端口成员类型不同，查询的结果也不同，需从对应的字段进行解析端口成员
    def get_ports_from_flv_members(self, member_type, flv_members):
        result_port_list = []
        if member_type == "eth_port":
            for member_info in flv_members:
                eth_port = member_info.get("ID")
                result_port_list.append(eth_port)
        elif member_type == "bond_port":
            for member_info in flv_members:
                bond_port_list = member_info.get("Port ID List")
                result_port_list.extend(bond_port_list.split(","))
        elif member_type == "vlan":
            for member_info in flv_members:
                vlan_port = member_info.get("Port ID")
                result_port_list.append(vlan_port)
        return result_port_list
