# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.product_util import compare_patch_version

PASS_VERSION_DICT = {
    "6.1.5": "SPH30",
    "6.1.6": "SPH7",
}


class CheckItem(BaseCheckItem):
    def execute(self):
        # 步骤1检查是否2P
        description = self.dsl("exec_cli 'show controller general' | regex 'Electronic Label["
                               "\\s\\S]+Description=([\\s\\S]+2P)'")
        if not description:
            return CheckStatus.PASS, ""

        # 步骤2检查高低可靠内存
        mem_info = self.dsl("exec_developer 'show system memory' | vertical_parser")
        if not mem_info:
            return CheckStatus.PASS, ""

        if mem_info[0].get("Management Policy", "Unknown") != "Normal":
            return CheckStatus.PASS, ""

        # 步骤3检查FE ROCE卡数量
        cards = self.dsl("exec_cli 'show interface_module' | horizontal_parser")
        if not cards:
            return CheckStatus.PASS, ""
        cnt = 0
        exist = False
        for card in cards:
            # 获取卡类型
            model = card.get("Model")
            # 存在FE RoCE卡
            if "FE" in model and "ROCE" in model:
                exist = True
            # 存在FE RoCE卡或者DTOE卡
            if "FE" in model and ("ROCE" in model or "ETH" in model):
                cnt += 1
        #  是否存在ROCE卡或超过8张卡
        if not exist or cnt < 8:
            return CheckStatus.PASS, ""
        # 步骤4获取补丁版本
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        base_version = version_info.get("base_version").get("Current Version")

        if re.compile(r"SPH(\d+)").match(patch_version):
            now_version = base_version + '.' + patch_version
        else:
            now_version = base_version
        err_msg = self.get_msg("check.not.pass", now_version)
        # 检查补丁版本
        if base_version in PASS_VERSION_DICT:
            if compare_patch_version(patch_version, PASS_VERSION_DICT[base_version]) >= 0:
                return CheckStatus.PASS, ""
            else:
                return CheckStatus.WARNING, err_msg

        return CheckStatus.PASS, ""
