# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import time

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.product_util import is_support_cross_engine_ssh

TEN = 10


class CheckItem(BaseCheckItem):
    def execute(self):
        # 是否支持跨引擎ssh连接
        if not is_support_cross_engine_ssh(self.context.dev_node.version, self.context.dev_node.model):
            return CheckStatus.PASS, ""

        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")

        normal_node_str = self.get_normal_node_str()
        if not normal_node_str:
            return CheckStatus.PASS, ""

        nodes_status_list = self.get_nodes_status_list()
        has_main_flag = self.check_has_main_flag(nodes_status_list)
        if not has_main_flag:
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass") + self.get_msg(
                "check.suggestion.nomain") + self.get_msg("item.suggestion")

        abnormal_nodes = self.get_abnomal_nods(normal_node_str, nodes_status_list)
        all_msgs = []
        if abnormal_nodes:
            for node in abnormal_nodes:
                all_msgs.append(self.get_msg("check.suggestion", node))

        if all_msgs:
            return CheckStatus.NOT_PASS, "\n".join(all_msgs) + self.get_msg("item.suggestion")

        return CheckStatus.PASS, ""

    # 获取当前接入节点
    def get_normal_node_str(self):
        dev_infos = self.dsl("exec_diagnose 'sys showcls' | vertical_parser")
        if dev_infos:
            dev_info = dev_infos[0]
            normal_node_bitmap = int(dev_info.get("normalNodeBitmap"))
        normal_node = bin(normal_node_bitmap)
        normal_str = normal_node[2:]
        if len(normal_node) >= TEN:
            normal_str = normal_node[-8:]
        normal_node_str = normal_str[::-1]

        return normal_node_str

    # 获取CCDB控制器状态
    def get_ccdb_controllers_status(self):
        status_info = self.dsl(
            "exec_mini_system 'ccdb.sh -c showstate' | splitlines | regex 'nid = (?P<nid>\d+) ,"
            "role = (?P<role>\d+) \(1:M,2:S,3:R\),status = (?P<status>\d+) \(1:R,2:N,3:AbN\)'",
            end_str_list=["minisystem>", "minisystem> "]
        )

        return status_info

    def get_nodes_status_list(self):
        status_results = self.dsl("exec_on_all {}", self.get_ccdb_controllers_status)
        nodes_status_list = []
        for node_id, status_info in status_results.items():
            if not status_info:
                continue
            for line in status_info:
                role_status = 'nid = ' + str(line.get("nid")) + ' ,' 'role = ' + str(
                    line.get("role")) + ' (1:M,2:S,3:R),status = ' + str(
                    line.get("status")) + ' (1:R,2:N,3:AbN)'
                nodes_status_list.append(role_status)
        return nodes_status_list

    # 判断是否有主
    def check_has_main(self, nodes_status_list):
        for info in nodes_status_list:
            if "role = 1" in info:
                return True
        return False

    def check_has_main_flag(self, nodes_status_list):
        has_main_flag = self.check_has_main(nodes_status_list)
        retry_times = 0
        while not has_main_flag and retry_times < 2:
            time.sleep(20)
            nodes_status_list = self.get_nodes_status_list()
            has_main_flag = self.check_has_main(nodes_status_list)
            retry_times += 1
        return has_main_flag

    def get_abnomal_nods(self, normal_node_str, nodes_status_list):
        """
        获取状态不正常的控制器
        :param normal_node_str: 获取的接入的节点
        :param nodes_status_list: 获取的CCDB控制器状态列表
        :return: 不正常的控制器列表
        """
        index = 0
        abnormal_nodes = []
        for node_str in normal_node_str:
            if node_str == '1':
                if "status = 3" in nodes_status_list[index]:
                    ab_node = 'nid = ' + str(index)
                    abnormal_nodes.append(ab_node)
            index += 1

        return abnormal_nodes

