# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.
import time
import ast

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.dsl.dsl_common import get_engine_height
from psdk.platform.util.product_util import get_node_id_by_ctrl_id
from psdk.platform.util.product_util import is_support_cross_engine_ssh
from psdk.platform.util.echo_parser import get_horizontal_cli_ret



class CheckItem(BaseCheckItem):
    def execute(self):
        # 是否支持跨引擎ssh连接
        if not is_support_cross_engine_ssh(self.context.dev_node.version, self.context.dev_node.model):
            return CheckStatus.PASS, ""

        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")

        # 接入节点
        normal_node_num_str = self.get_normal_node_num_str()
        if normal_node_num_str:
            return CheckStatus.PASS, ""

        nodes_status_list = self.get_nodes_status_list()

        retry_times = 0
        while not nodes_status_list and retry_times < 2:
            time.sleep(20)
            nodes_status_list = self.get_nodes_status_list()
            retry_times += 1
        if not nodes_status_list:
            return CheckStatus.NOT_PASS, self.get_msg("check.suggestion.other") + self.get_msg("item.suggestion")

        # 选中的控制器的nid
        fru_node_id = self.get_selected_fru_nid()

        normal_index, abnormal_nodes = self.get_abnomal_nods(nodes_status_list, fru_node_id)
        
        if normal_index >= 1:
            return CheckStatus.PASS, ""
        all_msgs = []
        if abnormal_nodes:
            for node in abnormal_nodes:
                all_msgs.append(self.get_msg("check.suggestion", node))

        if all_msgs:
            return CheckStatus.NOT_PASS, "\n".join(all_msgs) + self.get_msg("item.suggestion")

        return CheckStatus.PASS, ""

    # 获取待更换控制器的node id
    def get_selected_fru_nid(self):
        selected_fan = ast.literal_eval(self.context.execute_env.selected_fru)
        selected_fru_id = selected_fan[0].get("id", "").strip() if isinstance(selected_fan, list) else \
            selected_fan.get("id", "").strip()
        engine_height = get_engine_height(self.dsl, self.context)
        fru_node_id = get_node_id_by_ctrl_id(selected_fru_id, engine_height)

        return fru_node_id

    # 获取当前接入节点
    def get_normal_node_num_str(self):
        dev_infos = self.dsl("exec_diagnose 'sys showcls' | vertical_parser")
        if dev_infos:
            dev_info = dev_infos[0]
            normal_node_bitmap = int(dev_info.get("normalNodeBitmap"))
            return normal_node_bitmap == 0
        return False

    # 获取CCDB控制器状态
    def get_ccdb_controllers_status(self):
        status_info = self.dsl(
            "exec_mini_system 'ccdb.sh -c showdb ccdb_metadata.db'", end_str_list=["minisystem>", "minisystem> "]
        )
        return status_info

    # 截取CCDB状态列表
    def get_nodes_status_list(self):
        status_results = self.dsl("exec_on_all {}", self.get_ccdb_controllers_status)

        for _, status_info in status_results.items():
            if not status_info:
                continue
            status_info_find_index = status_info.find("node_id")
            if status_info_find_index == -1:
                continue
            start_index = status_info.find("node_id")
            end_index = status_info.find("src")
            nodes_status_list = get_horizontal_cli_ret(status_info[start_index:end_index])
            return nodes_status_list
        return []

    # 筛选normal和abnormal节点
    def get_abnomal_nods(self, nodes_status_list, fru_node_id):
        """
        获取状态不正常的控制器
        :param nodes_status_list: 获取的CCDB控制器状态列表
        :param fru_node_id: 待更换的备件的nid
        :return: 不正常的控制器列表
        """
        normal_index = 0
        abnormal_nodes = []

        for node_status in nodes_status_list:
            service_status = node_status["service_state"]
            session_status = node_status["session_state"]
            nid = node_status["node_id"]
            if service_status == "normal" and session_status == "normal":
                if fru_node_id == nid:
                    continue
                normal_index += 1
            else:
                abnormal_nodes.append("nid = " + str(nid))
        return normal_index, abnormal_nodes
