# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.echo_parser import get_horizontal_cli_ret_compatible_chinese
from psdk.dsl import fault_mode as ft
from psdk.dsl.common import DslException


class CheckItem(BaseCheckItem):
    @staticmethod
    def check_is_the_same_interface(port_id_set, interface_id_list):
        # 没有卡或超过两张卡就通过
        if not port_id_set or len(port_id_set) >= 2:
            return False
        for interface_id in interface_id_list:
            for port_id in port_id_set:
                if port_id in interface_id:
                    return True
        return False

    def execute(self):
        all_err_msg = []
        vxlan_interfaces = self.get_vxlan_interface()
        if not vxlan_interfaces:
            return CheckStatus.PASS, ""

        err_vxlan_logical_ports = self.get_logical_ports_on_interface(vxlan_interfaces)
        if err_vxlan_logical_ports:
            all_err_msg.append(err_vxlan_logical_ports)
        err_vxlan_vlan = self.get_vxlan_vlan(vxlan_interfaces)
        if err_vxlan_vlan:
            all_err_msg.append(err_vxlan_vlan)
        err_vxlan_bond_port = self.get_vxlan_bond_port(vxlan_interfaces)
        if err_vxlan_bond_port:
            all_err_msg.append(err_vxlan_bond_port)
        if all_err_msg:
            return CheckStatus.WARNING, "\n".join(all_err_msg)
        return CheckStatus.PASS, ""

    def get_vxlan_bond_port(self, vxlan_interfaces):
        vxlan_bond_port = []
        cli_ret = self.dsl("exec_cli 'show bond_port'")
        bond_port_info = get_horizontal_cli_ret_compatible_chinese(cli_ret)
        bond_ports = [bond_port for bond_port in bond_port_info if bond_port.get("Bond Port Type") == "Host"]
        if not bond_ports:
            return ""

        for bond_port in bond_ports:
            port_id_list = bond_port.get("Port ID List", "").split(",")
            port_id_set = set([".".join(port_id.split(".")[:-1]) for port_id in port_id_list])
            if self.check_is_the_same_interface(port_id_set, vxlan_interfaces):
                vxlan_bond_port.append(bond_port.get("Name"))
        return self.get_msg("check.not.pass.bondport", ", ".join(vxlan_bond_port)) if vxlan_bond_port else ""

    def get_vxlan_vlan(self, vxlan_interfaces):
        vxlan_vlan = []
        vlan_info = self.dsl("exec_cli 'show vlan general' | horizontal_parser")
        vlans = [vlan for vlan in vlan_info if vlan.get("Port Type") == "ETH"]
        if not vlans:
            return ""
        [
            vxlan_vlan.append(vlan.get("Name"))
            for vlan in vlans
            if ".".join(vlan.get("Port ID", "").split(".")[:-1]) in vxlan_interfaces
        ]
        return self.get_msg("check.not.pass.vlanport", ", ".join(vxlan_vlan)) if vxlan_vlan else ""

    def get_logical_ports_on_interface(self, vxlan_interfaces):
        cli_ret = self.dsl("exec_cli 'show logical_port general'")
        logical_port_info = get_horizontal_cli_ret_compatible_chinese(cli_ret)
        logical_ports = [
            logical_port
            for logical_port in logical_port_info
            if logical_port.get("Current Port Type") == "ETH"
        ]
        if not logical_ports:
            return ""
        vxlan_logical_ports = [
            logical_port.get("Logical Port Name")
            for logical_port in logical_ports
            if ".".join(logical_port.get("Current Port ID").split(".")[:-1]) in vxlan_interfaces
        ]
        return (
            self.get_msg("check.not.pass.logicalport", ", ".join(vxlan_logical_ports)) if vxlan_logical_ports else ""
        )

    def get_vxlan_interface(self):
        interface_info = self.dsl("exec_cli 'show interface_module' | horizontal_parser",
                                  return_if={ft.FindStr("Error:"): "noCheck"})
        if interface_info == "noCheck":
            raise DslException("query.result.abnormal")
        return [interface.get("ID") for interface in interface_info if interface.get("Usage Type") == "Cloud Vxlan"]
