# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl import fault_mode as ft


class CheckItem(BaseCheckItem):
    def execute(self):
        # 步骤2 执行命令：show file_system general|filterColumn include columnList=ID,Dedup\sEnabled,Compression\sEnabled,Block\sSize，记录所有文件系统的ID、Dedup Enabled、Compression Enabled、Block Size字段值；
        file_system_block = self.dsl(
            "exec_cli 'show file_system general|filterColumn include columnList=ID,Dedup\sEnabled,Compression\sEnabled,Block\sSize' | horizontal_parser ",
            return_if={ft.CmdNoSupport: "notSupport",},
        )
        err_msg_list = list()
        if file_system_block == "notSupport":
            return CheckStatus.PASS, ""
        error_file_system_list = self.check_file_system(file_system_block)
        if error_file_system_list:
            err_msg_list.append(
                self.get_msg(
                    "check.not.pass.file.system",
                    ",".join(error_file_system_list)
                )
            )

        # 步骤3 执行命令：show lun dedup_compress|filterColumn include columnList=ID，获取所有重删压缩LUN的ID；
        lun_dedup_compress = self.dsl(
            "exec_cli 'show lun dedup_compress|filterColumn include columnList=ID' | horizontal_parser "
        )
        if not lun_dedup_compress and not file_system_block:
            return CheckStatus.PASS, ""
        # 步骤4 执行命令：show lun general |filterColumn include columnList=ID,Cow\sGrain\sSize|filterRow column=Cow\sGrain\sSize predict=not predict2=equal_to value=16.00KB，记录块大小不为16KB的LUN ID。
        lun_16kb = self.dsl(
            "exec_cli 'show lun general |filterColumn include columnList=ID,Cow\sGrain\sSize|filterRow column=Cow\sGrain\sSize predict=not predict2=equal_to value=16.00KB' | horizontal_parser "
        )
        dedup_compress_id_list = self.get_id_list(lun_dedup_compress)
        lun_id_list = self.get_id_list(lun_16kb)
        err_id_list = list(set(dedup_compress_id_list) & set(lun_id_list))
        if err_id_list:
            err_msg_list.append(
                self.get_msg("check.not.pass.lun", ",".join(err_id_list))
            )
        if err_msg_list:
            return CheckStatus.NOT_PASS, "\n".join(err_msg_list)
        return CheckStatus.PASS, ""

    @staticmethod
    def check_file_system(file_system_list):
        """
        存在文件系统的“Dedup Enabled”或“Compression Enabled”字段值为“Yes”，
        且Block Size不为16KB
        """
        error_file_system_list = [
            file_system.get("ID")
            for file_system in file_system_list
            if (
                file_system.get("Dedup Enabled") == "Yes"
                or file_system.get("Compression Enabled") == "Yes"
            )
            and file_system.get("Block Size", "").replace(".000", "") != "16KB"
        ]
        return error_file_system_list

    @staticmethod
    def get_id_list(cli_ret_list):
        """
        获取ID
        :param cli_ret_list:
        :return:
        """
        return [cli_ret.get("ID") for cli_ret in cli_ret_list if cli_ret]
