# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import ast
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.protocol.cli_con_mgr import get_available_conn
from psdk.platform.protocol.cli_service import enter_developer_from_cli

FRU_TYPE = {
    "fru_controller": 'controller',
    "fru_controller_enclosure": 'controller_enclosure',
    "fru_disk_enclosure": 'disk_enclosure',
    "fru_ip_scale_out_interface": 'interface_card',
    "fru_back_end_interface": 'interface_card',
    "fru_front_end_interface": 'interface_card',
    "fru_other_interface": 'interface_card',
    "fru_pcie_interface": 'interface_card',
    "fru_disk": 'hard_disk',
    "fru_power": 'power_supply',
    "fru_bbu": 'bbu',
    "fru_xpu": 'controller',
    "fru_system_disk": 'controller'
}

# 端口类型与 type_name的对应关系
PORT_TYPE = {
    "FC": 'fibre_channel_port',
    "ETH": 'eth_iscsi_port',
    "RoCE": 'roce_port',
    "SAS": 'sas_port'
}


class CheckItem(BaseCheckItem):
    def execute(self):
        # 对于风扇模块，如果后续也要做该检查项，注意风扇模块可以多选，并且可能object_id参数中有空格
        selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
        self.logger.info("----select_fru:{}".format(selected_fru))
        if isinstance(selected_fru, list):
            selected_fru = selected_fru.get(0)
        object_id = self.get_object_id(selected_fru)
        type_name = self.get_type_name(object_id)
        if not type_name:
            return CheckStatus.PASS, ""
        delete_potential_risk_cmd = "delete potential_risk type_name={} object_id={}".format(type_name, object_id)
        from com.huawei.ism.tool.obase.exception import ToolException
        try:
            result = self.exec_delete_potential_risk_cmd(delete_potential_risk_cmd)
            if 'Command executed successfully' not in result:
                # 弹框提示，但仍然检查通过，不影响备件更换流程
                self.show_alarm_tip(self.get_msg("check.not.pass"))
            return CheckStatus.PASS, ""
        except (ToolException, Exception) as cmd_error:
            error_info = "the command for deleting historical faulty slot information fails: {}".format(cmd_error)
            self.logger.error(error_info)
            self.show_alarm_tip(self.get_msg("check.not.pass"))
            return CheckStatus.PASS, ""

    def get_object_id(self, selected_fru):
        object_id = str(selected_fru.get("location", ""))
        tool_name = self.context.execute_env.tool_name
        # 对于电源模块，去掉object_id参数中的空格
        if tool_name == "fru_power":
            object_id = object_id.replace(' ', '')
        if tool_name == "fru_controller_enclosure" or tool_name == "fru_disk_enclosure":
            object_id = str(selected_fru.get("name", ""))
        return object_id

    def exec_delete_potential_risk_cmd(self, delete_potential_risk_cmd):
        ret = ''
        cli = get_available_conn(self.context, cli=None, allow_not_admin=False)  # cli连接
        enter_developer_from_cli(cli, self.lang)
        ret += cli.execCmd(delete_potential_risk_cmd)
        if ret.strip().endswith('(y/n)') or ret.strip().endswith('[y/n]'):
            ret += cli.execCmd('y')
        return ret

    def get_type_name(self, object_id):
        """
        获取type_name的值

        :param object_id: location
        :return: type_name
        """
        type_name = ''
        tool_name = self.context.execute_env.tool_name
        if tool_name == "fru_exp_board":
            expansion_board_type = self.dsl(
                "exec_cli 'show expansion_module expansion_module_id={}' | vertical_parser | get_key('Type')".format(
                    object_id))
            if "SAS" in expansion_board_type:
                type_name = 'sas_expansion_board'
            else:
                type_name = 'ip_expansion_board'
        elif tool_name in FRU_TYPE:
            type_name = FRU_TYPE.get(tool_name, '')
        else:
            # 针对端口
            port_type_name = self.get_port_type_name(object_id)
            if port_type_name:
                type_name = port_type_name
        return type_name

    def get_port_type_name(self, selected_location):
        for key, value in PORT_TYPE.items():
            location_list = self.dsl(
                "exec_cli 'show port general physical_type={}' | horizontal_parser | get_key('ID')".format(key))
            if selected_location in location_list:
                return value
        return ''

    @staticmethod
    def show_alarm_tip(msg):
        from com.huawei.ism.tool.bizpack.wizardparse.intf import GetUserInputImpl
        dialog = GetUserInputImpl()
        dialog.showInfoDialog(msg)
