# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version


PASS_VERSION_DICT = {
    "V500R007C61": "V500R007C61SPH030",
    "V300R006C61": "V300R006C61SPH030"
}

SOLVE_VERSION_DICT = {
    "V500R007C61": "V500R007C61SPH056",
    "V300R006C61": "V300R006C61SPH056"
}


class CheckItem(BaseCheckItem):
    def execute(self):
        # 获取基本信息
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        his_patch_version = version_info.get("patch_version").get("History Version")
        base_version = version_info.get("base_version").get("Current Version")

        # 执行检查
        if base_version not in PASS_VERSION_DICT:
            return CheckStatus.PASS, ""

        if self.check_patch(base_version, patch_version, his_patch_version):
            return CheckStatus.PASS, ""

        if self.check_patch_solve(base_version, patch_version):
            return CheckStatus.PASS, ""

        result_info = self.dsl("exec_developer 'show interface_module' | regex '(4 port SmartIO I/O Module)'")
        if not result_info:
            return CheckStatus.PASS, ""
        
        effect_info = self.dsl(
            "exec_mini_system 'cat /startup_disk/image/boot/versions.conf' | regex '(Version=1\\.20T(13|18))'"
        )
        if effect_info:
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion", (patch_version, ))
        else:
            return CheckStatus.PASS, ""

    def check_patch(self, base_version, patch_version, his_patch_version):
        return (PASS_VERSION_DICT.get(base_version)) and \
        ((compare_patch_version(patch_version, PASS_VERSION_DICT[base_version]) >= 0) and
        compare_patch_version(his_patch_version, PASS_VERSION_DICT[base_version]) >= 0) or \
        ((compare_patch_version(patch_version, PASS_VERSION_DICT[base_version]) >= 0) and his_patch_version == '--')

    def check_patch_solve(self, base_version, patch_version):
        return (SOLVE_VERSION_DICT.get(base_version)) and \
        ((compare_patch_version(patch_version, SOLVE_VERSION_DICT[base_version]) >= 0) or \
        (compare_patch_version(patch_version, SOLVE_VERSION_DICT[base_version]) >= 0))