# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.product_util import compare_patch_version
from psdk.platform.util.base_util import get_common_msg
from psdk.dsl.dsl_common import get_version_info
from psdk.dsl import fault_mode as ft

CHECK_MIRROR_BOND_PASS_VERSION = {'6.1.2': 'SPH12',
                                  'V500R007C71SPC100': 'SPH115',
                                  'V500R007C73SPC100': 'SPH105'}


class CheckItem(BaseCheckItem):
    def execute(self):
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        version = version_info.get("base_version").get("Current Version")
        for soft, patch in CHECK_MIRROR_BOND_PASS_VERSION.items():
            if version == soft and \
                    compare_patch_version(patch_version, patch) >= 0:
                return CheckStatus.PASS, ""

        if version == "6.1.2":
            return self.check_is_have_multi_bond_for_v6()

        return self.check_is_have_multi_bond_for_v5()

    def check_bond_member_is_multi(self, bond_mem):
        """
        检查是否是跨卡bond
        """
        port_name_s = bond_mem.lstrip().split(",")
        card0 = port_name_s[0].lstrip().split(".", 3)
        for line in port_name_s:
            card1 = line.lstrip().split(".", 3)
            if card0[1] == card1[1] and card0[2] == card1[2]:
                continue
            return True
        return False

    def check_is_have_bond(self, cur_port_id):
        bond_port_info = self.dsl("exec_cli 'show bond_port' | horizontal_parser")
        if 'No matching records' in bond_port_info:
            return False
        for bond_port in bond_port_info:
            if bond_port.get("Name") != cur_port_id:
                continue
            port_list = bond_port.get("Port ID List")
            res = self.check_bond_member_is_multi(port_list)
            if res:
                return True
        return False

    def check_lif_is_link_down(self, link_status, v_stores):
        """
        融合 检查lif状态是否是link down
        """
        logical_ports = \
            self.dsl("exec_cli 'show logical_port general' | horizontal_parser",
                     return_if={ft.FindStr("Error:"): "not_check"})
        if logical_ports == "not_check":
            return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")
        for logical_port in logical_ports:
            if logical_port.get("Current Port Type") != "Bond" or logical_port.get("Running Status") != link_status:
                continue
            if v_stores:
                lif_v_store = logical_port.get("vStore ID")
                if lif_v_store not in v_stores:
                    continue
            cur_port_id = logical_port.get("Current Port ID")
            res = self.check_is_have_bond(cur_port_id)
            if res:
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")

        return CheckStatus.PASS, ""

    def check_is_have_multi_bond_for_v6(self):
        return self.check_lif_is_link_down("Standby", [])

    def check_is_have_multi_bond_for_v5(self):
        v_store_pairs = \
            self.dsl("exec_cli 'show hyper_metro_vstore_pair general' | horizontal_parser",
                     return_if={ft.FindStr("Error:"): "not_check"})
        if v_store_pairs == "not_check":
            return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")
        v_stores = []
        for v_store_pair in v_store_pairs:
            if v_store_pair.get("Active/Passive") != "Passive":
                continue
            v_store_id = v_store_pair.get("Local Vstore ID")
            v_stores.append(v_store_id)
        if not v_stores:
            return CheckStatus.PASS, ""
        return self.check_lif_is_link_down("Link Down", v_stores)
