# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.echo_parser import get_horizontal_cli_ret


PRODUCT_MIN_VERSION_MAP = {
    '6.0.1': 50,
    '6.1.0': 30,
    '6.1.2': 36,
    '6.1.3': 16,
    '6.1.5': 12,
}

PRODUCT_MAX_VERSION_MAP = {
    '6.0.1': 50,
    '6.1.0': 31,
    '6.1.2': 36,
    '6.1.3': 25,
    '6.1.5': 16,
}


class CheckItem(BaseCheckItem):

    @staticmethod
    def get_system_version_without_ret(cli_ret):
        product_info = []
        hot_patch_begin_index = cli_ret.find("HotPatch Version")
        software_begin_index = cli_ret.find("Software Version")

        if software_begin_index == -1 or hot_patch_begin_index == -1:
            return False, product_info
        software_version = get_horizontal_cli_ret(cli_ret[software_begin_index:hot_patch_begin_index])[0]. \
            get("Current Version")
        controller_names = [d.get('Name')
                            for d in get_horizontal_cli_ret(cli_ret[software_begin_index:hot_patch_begin_index])]
        hot_patch_version = get_horizontal_cli_ret(cli_ret[hot_patch_begin_index:])[0].get("Current Version")
        if hot_patch_version == "--":
            hot_patch_version = "0"
        product_info.append(software_version)
        product_info.append(hot_patch_version)
        product_info.append(controller_names)
        return True, product_info

    @staticmethod
    def check_product_version(software_version, hot_patch_version):
        if int(re.search(r'\d+', hot_patch_version).group()) < PRODUCT_MIN_VERSION_MAP.get(software_version):
            return True
        return int(re.search(r'\d+', hot_patch_version).group()) > PRODUCT_MAX_VERSION_MAP.get(software_version)

    @staticmethod
    def get_ssd_dd_engine_info(ssd_disk_domain_infos, engine_map):
        for ssd_disk_domains_info in ssd_disk_domain_infos:
            engine_list = ssd_disk_domains_info.get('Controller enclosure', '').split(',')
            for key in engine_list:
                if not engine_map.get(key):
                    engine_map[key] = 0
                engine_map[key] += 1
        return engine_map

    def get_disk_domain_info(self):
        cli_ret = self.dsl("exec_cli 'show disk_domain general |"
                           "filterColumn include columnList=ID' | horizontal_parser")
        disk_domains = [item.get("ID") for item in cli_ret]
        return len(disk_domains) < 2, disk_domains

    def check_single_node_switch(self):
        """
        检查单控制器的冲突链开关状态，若打开，则返回False，否则返回True
        """
        switch_infos = self.dsl("exec_diagnose 'ddp set setDdpReadIoCollisionSwitch status' | splitlines")
        for switch_info in switch_infos:
            if "1" in switch_info:
                return False
        return True

    def check_all_controller_switch_status(self):
        """
        检查所有控制器的冲突链开关，若有一个返回False（即有一个控制器的冲突链为打开状态），则返回False
        """
        result_info = self.dsl("exec_on_all {}", self.check_single_node_switch)
        return all(result_info.values())

    def get_all_ssd_disk_domain(self, disk_domains):
        ssd_disk_domains = []
        for disk_domain in disk_domains:
            disk_domains_infos = self.dsl("exec_diagnose 'dop showDataFlowParam {}' |"
                                          " vertical_parser".format(disk_domain))
            for disk_domains_info in disk_domains_infos:
                if int(disk_domains_info.get("is have capacity tier")) < 1:
                    ssd_disk_domains.append(disk_domain)
        return len(ssd_disk_domains) < 2, ssd_disk_domains

    def check_ssd_engine(self, ssd_disk_domains):
        engine_map = {}
        for ssd_disk_domain in ssd_disk_domains:
            ssd_disk_domain_infos = self.dsl("exec_cli 'show disk_domain general disk_domain_id={}' | vertical_parser"
                                             .format(ssd_disk_domain))
            engine_map = self.get_ssd_dd_engine_info(ssd_disk_domain_infos, engine_map)
        return not any(engine_map.get(engine_id) > 1 for engine_id in engine_map)

    def check_ddp_read_io(self, controller_names):
        for controller_name in controller_names:
            ddp_infos = self.dsl("exec_cli 'show ddp io_counter controller={}' "
                                 "| horizontal_parser".format(controller_name))
            for ddp_info in ddp_infos:
                if ddp_info.get('Phase Name') == 'DDP_READ_IO' and int(ddp_info.get('Not Back Num')) > 0:
                    return False
        return True

    def execute(self):
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")
        # 检查方法步骤1：获取产品版本信息，非指定版本，检查通过
        # 获取产品信息
        flag, product_info = \
            self.get_system_version_without_ret(self.dsl("exec_cli 'show upgrade package'"))
        if not product_info:
            return CheckStatus.PASS, ""

        software_version, hot_patch_version, controller_names = product_info

        if software_version not in PRODUCT_MIN_VERSION_MAP:
            return CheckStatus.PASS, ""
        # 检查是否为风险版本
        if self.check_product_version(software_version, hot_patch_version):
            return CheckStatus.PASS, ""

        # 检查方法步骤2：获取硬盘域信息，硬盘域个数小于2，检查通过
        disk_domain_check_res, disk_domains = self.get_disk_domain_info()
        if disk_domain_check_res:
            return CheckStatus.PASS, ""

        # 检查方法步骤3：遍历所有控制器，读相似指纹开关都为关闭，检查通过
        if self.check_all_controller_switch_status():
            return CheckStatus.PASS, ""

        # 检查方法步骤4：识别是否为全闪硬盘域，若全闪硬盘域个数小于2，检查通过
        if "Dorado" in self.context.dev_node.model:
            ssd_disk_domains = disk_domains
        else:
            all_ssd_disk_domain_res, ssd_disk_domains = self.get_all_ssd_disk_domain(disk_domains)
            if all_ssd_disk_domain_res:
                return CheckStatus.PASS, ""

        # 检查方法步骤5：所有全闪硬盘域都归属于不同的引擎，检查通过
        if self.check_ssd_engine(ssd_disk_domains):
            return CheckStatus.PASS, ""

        # 检查方法步骤6：不存在DDP_READ_IO计数或DDP_READ_IO的notBack为0，检查通过
        if self.check_ddp_read_io(controller_names):
            return CheckStatus.PASS, ""

        product_version = software_version + ".SPH" + hot_patch_version
        return CheckStatus.NOT_PASS, self.get_msg("check.not.pass", product_version)
