# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2013-2024. All rights reserved.

import re
import time

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl import fault_mode as ft
from psdk.platform.adapter import java_connector_factory as connector



class CheckItem(BaseCheckItem):
    def execute(self):
        current_ip = self.context.get_dev_node().ip
        current_sn = self.context.get_dev_node().sn
        logic_ports = \
            self.dsl(
                "exec_cli 'show port general logic_type=Management_Port physical_type=ETH running_status=link_up'"
                "| horizontal_parser",
                return_if={ft.FindStr("Error:"): "not_check"})
        if logic_ports == "not_check":
            return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")
        all_ip_list = []
        for line in logic_ports:
            temp_ip = line.get("IPv4 Address")
            if temp_ip != "--" and temp_ip != current_ip:
                port_host_ipv4 = line.get('IPv4 Address')
                all_ip_list.append(port_host_ipv4)

            temp6_ip = line.get("IPv6 Address")
            if temp6_ip != "--" and temp6_ip != current_ip:
                port_host_ipv6 = line.get('IPv6 Address')
                all_ip_list.append(port_host_ipv6)
        if not all_ip_list:
            return CheckStatus.PASS, ""

        err_msg_list = []
        for check_ip in all_ip_list:
            is_same, msg = self.check_mgt_ip(check_ip, current_sn)
            if is_same is False:
                err_msg_list.append(msg)
            elif is_same == CheckStatus.WARNING:
                err_msg_list.append(msg)

        if err_msg_list:
            return CheckStatus.WARNING, "\n".join(err_msg_list)
        return CheckStatus.PASS, ""

    def check_mgt_ip(self, ip, check_sn):
        sn_regex = re.compile(r"SN\s+:\s*(\w+)", re.IGNORECASE)
        new_dev_node = self.get_java_dev(ip)
        cli = connector.get_cli_con(new_dev_node)
        if not cli:
            return CheckStatus.WARNING, self.get_msg('check.not.pass')
        ret = cli.execCmd("show system general")
        cli.close()
        res_list = sn_regex.findall(ret)
        if not res_list:
            return CheckStatus.WARNING, self.get_msg('check.warning')
        sn = res_list[0]
        return sn == check_sn, self.get_msg('sn.not.check', sn)


    def get_java_dev(self, ip):
        """
        @summary: 将连接设备的信息封装为java的DevNode对象
        """
        import com.huawei.ism.tool.framework.pubservice.entity.DevNode as DevNode
        dev = self.context.get_java_dev()
        javadev = DevNode()
        javadev.setPort(dev.getPort())
        javadev.setSshForwardList(dev.getSshForwardList())
        javadev.setLoginUser(dev.getLoginUser())
        devicesn = dev.getDeviceSerialNumber()
        devicesn = devicesn + str(time.time())
        javadev.setDeviceSerialNumber(devicesn)
        javadev.setIp(ip)
        return javadev


