# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import string

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version
from psdk.dsl import fault_mode as ft


class CheckItem(BaseCheckItem):
    def execute(self):
        # 查询系统中接入的硬盘框
        enclosures_info = self.dsl("exec_cli 'show enclosure' | horizontal_parser",
                                   return_if={ft.FindStr("Error:"): "not_check"})
        if enclosures_info == "not_check":
            self.logger.info("Failed to obtain enclosures info. check failed.")
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.enclosure_info")
        # 检查查询的硬盘框信息中是否包含有NVMe JBOF硬盘框,如果没有则认为系统未接入该种硬盘框,检查项返回不涉及
        if "SSD NVMe disk enclosure" not in str(enclosures_info):
            self.logger.info("no NVMe JBOF disk enclosures. check result is Not involved.")
            return CheckStatus.NOT_SUPPORT, ""

        # 查询当前阵列产品版本
        version_info = get_version_info(self.dsl)
        if not version_info:
            self.logger.info("Failed to obtain version info. check failed")
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.version")
        # 获取产品版本和补丁版本,若获取不到产品版本和补丁版本则返回不通过
        sph_version = version_info.get("patch_version").get("Current Version")
        base_version = version_info.get("base_version").get("Current Version")
        # 判断当前版本是否满足：
        # V300R002C10SPC100及补丁版本大于等于V300R002C10SPH150或者V300R002C20及补丁版本大于等于V300R002C20SPH010,满足则直接默认通过
        pass_version_dict = {
            "V300R002C10SPC100": "V300R002C10SPH150",
            "V300R002C20": "V300R002C20SPH010"
        }
        if compare_patch_version(sph_version, pass_version_dict.get(base_version)) >= 0:
            self.logger.info("The baseline version and patch version meet the requirements, check pass")
            return CheckStatus.PASS, ""
        else:
            log = "The patch_ver is {}, which does not meet the requirements, check not pass".format(sph_version)
            self.logger.info(log)
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.result")
