# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version


class CheckItem(BaseCheckItem):
    def execute(self):
        mem_version = self.context.dev_node.version
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        base_version = version_info.get("base_version").get("Current Version")
        res, msg = self.check_qemu_kvm_leak_error(base_version, mem_version, patch_version)
        self.logger.info("res:{}, msg:{}".format(res, msg))
        return res, msg

    def check_qemu_kvm_leak_error(self, base_version, mem_version, patch_version):
        if str(base_version).startswith("V3") and base_version < "V300R006C60":
            if not self.has_vm():
                self.logger.info("smaller than v3r6c60, and no vm, Result:pass.")
                return CheckStatus.PASS, ""
            self.logger.info("smaller than v3r6c60, and has vm, Result:not pass.")
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.upd.v3r6", base_version, patch_version)
        if all(
                [
                    str(base_version).startswith("V3"),
                    base_version == "V300R006C60",
                    compare_patch_version(patch_version, "V300R006C60SPH005") < 0,
                ]
        ):
            if not self.has_vm():
                self.logger.info("smaller than V300R006C60SPH005, and no vm, Result:pass.")
                return CheckStatus.PASS, ""
            self.logger.info("smaller than V300R006C60SPH005, and has vm, Result:not pass.")
            return (
                CheckStatus.NOT_PASS,
                self.get_msg("check.not.pass", base_version, patch_version, "V300R006C60SPH005"),
            )
        return self.check_risk_version(base_version, mem_version, patch_version)

    def check_risk_version(self, base_version, mem_version, patch_version):
        if all([str(base_version).startswith("V5"), "Kunpeng" not in mem_version, base_version < "V500R007C61"]):
            if not self.has_vm():
                self.logger.info("smaller than V500R007C61, and no vm, Result:pass.")
                return CheckStatus.PASS, ""
            self.logger.info("smaller than V500R007C61, and has vm, Result:not pass.")
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.upd.v5r7c61", base_version, patch_version)

        if all(
            [
                str(base_version).startswith("V5"),
                "Kunpeng" not in mem_version,
                base_version == "V500R007C61",
                compare_patch_version(patch_version, "V500R007C61SPH008") < 0,
            ]
        ):
            if not self.has_vm():
                self.logger.info("V500R007C61, and no vm, Result:pass.")
                return CheckStatus.PASS, ""
            self.logger.info("V500R007C61SPH008, and has vm, Result:not pass.")
            return (
                CheckStatus.NOT_PASS,
                self.get_msg("check.not.pass", base_version, patch_version, "V500R007C61SPH008"),
            )
        return self.check_kunpeng_risk_version(base_version, mem_version, patch_version)

    def check_kunpeng_risk_version(self, base_version, mem_version, patch_version):
        if all(
                [
                    str(base_version).startswith("V5"),
                    "Kunpeng" in mem_version,
                    base_version == "V500R007C70",
                    compare_patch_version(patch_version, "V500R007C70SPH005") < 0,
                ]
        ):
            if not self.has_vm():
                self.logger.info("smaller than V500R007C70SPH005, and no vm, Result:pass.")
                return CheckStatus.PASS, ""
            self.logger.info("smaller than V500R007C70SPH005, and has vm, Result:not pass.")
            return (
                CheckStatus.NOT_PASS,
                self.get_msg("check.not.pass", base_version + " Kunpeng", patch_version, "V500R007C70SPH005"),
            )
        self.logger.info("version:{}, patch:{} Result:pass.".format(base_version, patch_version))
        return CheckStatus.PASS, ""

    def has_vm(self):
        return self.dsl("exec_cli 'show virtual_machine general'| horizontal_parser")
