# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg

SSD_TIER = '0'
SAS_TIER = '1'
NLSAS_TIER = '2'
SSD_RAID6_ZONE = '9'
SAS_RAID6_ZONE = '16395'
NLSAS_RAID6_ZONE = '32781'
SSD_TYPE_TUP = ('SSD', 'SSD SED', 'NVMe SSD', 'NVMe SSD SED')
SAS_TYPE_TUP = ('SAS', 'SAS SED')
NLSAS_TYPE_TUP = ('NearLine SAS', 'NearLine SAS SED', 'SATA', 'SATA SED')
MIN_RAID6_DISK_NUM = 4


class CheckItem(BaseCheckItem):
    dd_list = []
    expand_list = []
    is_exist_fs = True

    @staticmethod
    def get_disk_tier_type(disk_type):
        """
        查询硬盘类型对应的层级
        @param disk_type: 硬盘类型
        @return: ssd-SSD层 sas-SAS层 nlsas-NLSAS层
        """
        if disk_type in SSD_TYPE_TUP:
            return 'ssd'
        elif disk_type in SAS_TYPE_TUP:
            return 'sas'
        elif disk_type in NLSAS_TYPE_TUP:
            return 'nlsas'
        else:
            return None

    def execute(self):
        self.dd_list = []
        self.expand_list = []
        tool_name = self.context.execute_env.tool_name
        if tool_name == "exp_ctrl":
            return self.check_raid6_zone_expand_ctrl()
        if tool_name == "exp_dd" or tool_name == "exp_disk":
            return self.check_raid6_zone_expand_dd_disk()
        return self.check_raid6_zone_inspect()

    def check_raid6_zone_expand_ctrl(self):
        # 扩控场景，先获取硬盘扩容信息
        self.get_expand_disk_info()
        if not self.expand_list:
            return CheckStatus.PASS, ""
        err_info_list = []
        engine_infos = self.get_logic_engine_info()
        # 先对已有引擎检查
        result = self.check_raid6_every_engine(engine_infos, err_info_list)
        if not result:
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")
        dd_raid6_dict = self.dsl("exec_on_all{}", self.get_exp_dd_raid6_mem_disk, target_ctrl="master")
        if not dd_raid6_dict:
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")
        for expand_info in self.expand_list:
            engine_id = expand_info.get("engine")
            dd_id = expand_info.get("dd")
            if engine_id in engine_infos.keys():
                continue
            # 扩容盘数
            ssd_exp_disk_num = expand_info.get("ssd", 0)
            sas_exp_disk_num = expand_info.get("sas", 0)
            nlsas_exp_disk_num = expand_info.get("nlsas", 0)
            # raid6 zone成员盘数
            dd_raid6_info = [value.get(dd_id) for key, value in dd_raid6_dict.items()][0]
            if not dd_raid6_info:
                break
            ssd_raid6 = dd_raid6_info[0]
            sas_raid6 = dd_raid6_info[1]
            nlsas_raid6 = dd_raid6_info[2]
            # 检查扩容盘数是否大于raid6 zone成员盘数
            if (ssd_exp_disk_num > 0) and (ssd_exp_disk_num <= ssd_raid6):
                err_info_list.append(self.get_msg("check.not.pass.ssd", engine_id, dd_id))
            if (sas_exp_disk_num > 0) and (sas_exp_disk_num <= sas_raid6):
                err_info_list.append(self.get_msg("check.not.pass.sas", engine_id, dd_id))
            if (nlsas_exp_disk_num > 0) and (nlsas_exp_disk_num <= nlsas_raid6):
                err_info_list.append(self.get_msg("check.not.pass.nlsas", engine_id, dd_id))
        if err_info_list:
            return CheckStatus.WARNING, "\n".join(err_info_list)
        return CheckStatus.PASS, ""

    def check_raid6_zone_expand_dd_disk(self):
        err_msg_list = []
        # 扩容场景，先获取硬盘扩容信息
        self.get_expand_disk_info()
        if not self.expand_list:
            return CheckStatus.PASS, ""
        engine_infos = self.get_logic_engine_info()
        result = self.check_raid6_every_engine(engine_infos, err_msg_list)
        if not result:
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")
        if err_msg_list:
            return CheckStatus.WARNING, "\n".join(err_msg_list)
        return CheckStatus.PASS, ""

    def check_raid6_every_engine(self, logic_engine_infos, err_msg_list):
        for logic_engine in logic_engine_infos.keys():
            is_found = False
            # 检查逻辑引擎上是否有扩容
            for expand_info in self.expand_list:
                if expand_info.get("engine") == logic_engine:
                    is_found = True
                    break
            if not is_found:
                continue
            logic_engine_info = logic_engine_infos.get(logic_engine)
            # 物理引擎id以及物理引擎内的节点id
            errors = self.dsl("exec_on_all{}",
                              self.check_raid6_zone_exp_disk,
                              target_ctrl={logic_engine_info[0]: [logic_engine_info[1]]})
            if not errors:
                return False
            for error in errors.values():
                if error:
                    err_msg_list.extend(error)
        return True

    def check_raid6_zone_exp_disk(self):
        err_info_list = []
        cur_engine_id = self.get_current_logic_engine_id()
        for exp in self.expand_list:
            engine_id = exp.get("engine")
            if engine_id != cur_engine_id:
                continue
            # 检查硬盘域是否支持raid6
            dd_id = exp.get("dd")
            is_support_raid6 = self.check_support_raid6_zone(dd_id)
            if not is_support_raid6:
                continue
            # 查询各层级raid6成员盘数
            ssd_raid6 = self.get_raid6_mem_disk_num(dd_id, SSD_TIER, False)
            sas_raid6 = self.get_raid6_mem_disk_num(dd_id, SAS_TIER, False)
            nlsas_raid6 = self.get_raid6_mem_disk_num(dd_id, NLSAS_TIER, False)
            # 查询各层级已有盘数
            ssd_disk_num, sas_disk_num, nlsas_disk_num = self.get_tier_exist_disk_num(dd_id)
            # 查询各层级扩容盘数
            ssd_exp_disk_num = exp.get("ssd", 0)
            sas_exp_disk_num = exp.get("sas", 0)
            nlsas_exp_disk_num = exp.get("nlsas", 0)
            # 扩容后盘数大于raid6成员盘数，检查通过，否则不通过
            if (ssd_exp_disk_num > 0) and (ssd_disk_num + ssd_exp_disk_num) <= ssd_raid6:
                err_info_list.append(self.get_msg("check.not.pass.ssd", cur_engine_id, dd_id))
            if (sas_exp_disk_num > 0) and (sas_disk_num + sas_exp_disk_num) <= sas_raid6:
                err_info_list.append(self.get_msg("check.not.pass.sas", cur_engine_id, dd_id))
            if (nlsas_exp_disk_num > 0) and (nlsas_disk_num + nlsas_exp_disk_num) <= nlsas_raid6:
                err_info_list.append(self.get_msg("check.not.pass.nlsas", cur_engine_id, dd_id))
        return err_info_list

    def check_raid6_zone_inspect(self):
        # 巡检场景，获取所有硬盘域id
        err_list = []
        self.get_all_disk_domain_id()
        self.is_exist_fs = self.is_exist_fs_row_lun()
        # 获取逻辑引擎id，以及状态正常的节点id
        engine_infos = self.get_logic_engine_info()
        for engine_info in engine_infos.values():
            err_infos = self.dsl("exec_on_all{}",
                                 self.check_raid6_zone_ins_dd,
                                 target_ctrl={engine_info[0]: [engine_info[1]]})
            if not err_infos:
                return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")
            for err_info in err_infos.values():
                if err_info:
                    err_list.extend(err_info)
        if err_list:
            return CheckStatus.WARNING, "\n".join(err_list)
        return CheckStatus.PASS, ""

    def check_raid6_zone_ins_dd(self):
        err_info_list = []
        cur_engine_id = self.get_current_logic_engine_id()
        for dd in self.dd_list:
            # 检查硬盘域是否支持raid6
            is_support_raid6 = self.check_support_raid6_zone(dd)
            if not is_support_raid6:
                continue
            # 查询各层级已有盘数
            ssd_disk_num, sas_disk_num, nlsas_disk_num = self.get_tier_exist_disk_num(dd)
            # 查询各层级raid6成员盘数
            ssd_raid6 = self.get_raid6_mem_disk_num(dd, SSD_TIER, True)
            sas_raid6 = self.get_raid6_mem_disk_num(dd, SAS_TIER, True)
            nlsas_raid6 = self.get_raid6_mem_disk_num(dd, NLSAS_TIER, True)
            # 盘数大于raid6成员盘数，检查通过，否则不通过
            if (ssd_disk_num > 0) and (ssd_disk_num <= ssd_raid6):
                err_info_list.append(self.get_msg("check.not.pass.ssd", cur_engine_id, dd))
            if (sas_disk_num > 0) and (sas_disk_num <= sas_raid6):
                err_info_list.append(self.get_msg("check.not.pass.sas", cur_engine_id, dd))
            if (nlsas_disk_num > 0) and (nlsas_disk_num <= nlsas_raid6):
                err_info_list.append(self.get_msg("check.not.pass.nlsas", cur_engine_id, dd))
        return err_info_list

    def get_all_disk_domain_id(self):
        cmd = "exec_cli 'show disk_domain general|filterColumn include columnList=ID' | horizontal_parser"
        dd_infos = self.dsl(cmd)
        # 统计逻辑引擎个数，并为每个逻辑引擎获取一个状态正常的节点id
        for dd_info in dd_infos:
            self.dd_list.append(dd_info.get('ID'))
        return

    def is_exist_fs_row_lun(self):
        cmd = "exec_cli 'show file_system general|filterColumn include columnList=ID' | horizontal_parser"
        fs_infos = self.dsl(cmd)
        for fs_info in fs_infos:
            if fs_info:
                return True
        cmd = "exec_cli 'show lun dedup_compress|filterColumn include columnList=ID' | horizontal_parser"
        row_lun_infos = self.dsl(cmd)
        for row_lun_info in row_lun_infos:
            if row_lun_info:
                return True
        return False

    def get_expand_disk_info(self):
        exp_disk_list = self.context.execute_env.ori_env.get("exp_disk_list")
        if not exp_disk_list:
            exp_disk_list = self.context.execute_env.ori_env.get("expDiskList")
        for exp_disk in exp_disk_list:
            # 新建硬盘域场景，硬盘域id为空，这种场景不检查
            if not exp_disk.get("diskDomain"):
                continue
            is_found = False
            ssd_num = 0
            sas_num = 0
            nlsas_num = 0
            # 逻辑引擎id
            engine = exp_disk.get("logicEng")
            dd_id = exp_disk.get("diskDomain")
            tier_type = self.get_disk_tier_type(exp_disk.get("diskModel"))
            disk_num = int(exp_disk.get("diskNum"))
            if tier_type == 'ssd':
                ssd_num = disk_num
            elif tier_type == 'sas':
                sas_num = disk_num
            elif tier_type == 'nlsas':
                nlsas_num = disk_num
            else:
                continue
            for exp in self.expand_list:
                if engine == exp.get("engine") and dd_id == exp.get("dd"):
                    is_found = True
                    ssd_num += exp.get("ssd", 0)
                    sas_num += exp.get("sas", 0)
                    nlsas_num += exp.get("nlsas", 0)
                    exp.update({"engine": engine, "dd": dd_id, "ssd": ssd_num, "sas": sas_num, "nlsas": nlsas_num})
                    break
            if not is_found:
                self.expand_list.append(
                    {"engine": engine, "dd": dd_id, "ssd": ssd_num, "sas": sas_num, "nlsas": nlsas_num})

    def get_logic_engine_info(self):
        group_dict = {}
        node_infos = self.dsl("exec_diagnose 'sys showcls' | horizontal_parser")
        # 统计逻辑引擎个数，并为每个逻辑引擎获取一个状态正常的节点id
        if node_infos:
            for node_info in node_infos:
                node_id = node_info.get("id")
                group_id = node_info.get("group")
                engine_id = node_info.get("engine")
                if group_id not in group_dict:
                    group_dict[group_id] = [engine_id, node_id]
        return group_dict

    def get_current_logic_engine_id(self):
        dev_infos = self.dsl("exec_diagnose 'sys showcls' | vertical_parser")
        if dev_infos:
            dev_info = dev_infos[0]
            local_id = dev_info.get("local node id")
            node_infos = self.dsl("exec_diagnose 'sys showcls' | horizontal_parser")
            for node_info in node_infos:
                if node_info.get("id") == local_id:
                    return node_info.get("group")
        return None

    def check_support_raid6_zone(self, dd_id):
        cmd = "exec_diagnose 'spa dumpobjinfo -p {} -t 1 -o {}' | vertical_parser".format(dd_id, dd_id)
        dd_infos = self.dsl(cmd)
        if not dd_infos:
            return False
        return dd_infos[0].get("fsMetaUseRatioType") != "0"

    def get_exp_dd_raid6_mem_disk(self):
        dd_raid6_dict = {}
        for exp in self.expand_list:
            dd_id = exp.get("dd")
            if dd_id in dd_raid6_dict:
                continue
            # 检查硬盘域是否支持raid6
            is_support_raid6 = self.check_support_raid6_zone(dd_id)
            if not is_support_raid6:
                continue
            ssd_raid6_num = self.get_raid6_mem_disk_num(dd_id, SSD_TIER, False)
            sas_raid6_num = self.get_raid6_mem_disk_num(dd_id, SAS_TIER, False)
            nlsas_raid6_num = self.get_raid6_mem_disk_num(dd_id, NLSAS_TIER, False)
            dd_raid6_dict[dd_id] = [ssd_raid6_num, sas_raid6_num, nlsas_raid6_num]
        return dd_raid6_dict

    def get_raid6_mem_disk_num(self, dd_id, tier_id, is_inspect):
        cmd = "exec_diagnose 'spa queryzoneintier -p {} -t {}' | horizontal_parser".format(dd_id, tier_id)
        zone_infos = self.dsl(cmd)
        for zone_info in zone_infos:
            zone_id = zone_info.get("Zone ID").split('(')[0]
            is_ssd_zone_valid = (tier_id == SSD_TIER) and (zone_id == SSD_RAID6_ZONE)
            is_sas_zone_valid = (tier_id == SAS_TIER) and (zone_id == SAS_RAID6_ZONE)
            is_nlsas_zone_valid = (tier_id == NLSAS_TIER) and (zone_id == NLSAS_RAID6_ZONE)
            if is_ssd_zone_valid or is_sas_zone_valid or is_nlsas_zone_valid:
                cmd = "exec_diagnose 'spa dumpobjinfo -p {} -t 4 -o {}' | vertical_parser".format(dd_id, zone_id)
                tier_infos = self.dsl(cmd)
                if tier_infos:
                    tier_info = tier_infos[0]
                    return int(tier_info.get("memberDiskNum"))
        # 巡检场景，若zone不存在，且没有fs，不会引起重构失败和raid6元数据分配失败问题，因此返回0，让巡检通过
        if is_inspect and not self.is_exist_fs:
            return 0
        return MIN_RAID6_DISK_NUM

    def get_tier_exist_disk_num(self, dd_id):
        ssd_num = 0
        sas_num = 0
        nlsas_num = 0
        cmd = "exec_diagnose 'pmgr showdisk -p {}' | splitlines".format(dd_id)
        disk_infos = self.dsl(cmd)
        for disk_info in disk_infos:
            if "TYPE_SSD" in disk_info:
                ssd_num += 1
            elif "TYPE_SAS" in disk_info:
                sas_num += 1
            elif "TYPE_SATA" in disk_info:
                nlsas_num += 1
        return ssd_num, sas_num, nlsas_num
