# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import time
import re
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg

MDS_MODULE_ID = 301

WATER_MARK_THRESHOLD = 90  # 水位阈值为90

RETURN_CONTINUE = 0
RETURN_PASS = 1
RETURN_NO_PASS = -1

CHECK_CNT = 3
CHECK_RESULT = RETURN_PASS
MAX_LOOP_CNT = 10000


class CheckItem(BaseCheckItem):
    def execute(self):
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")
        result_info = self.dsl("exec_on_all {}", self.check_single_node)
        for result in result_info.values():
            if result == RETURN_NO_PASS:
                return CheckStatus.NOT_PASS, self.get_msg("check.no.pass")
        return CheckStatus.PASS, ""

    def check_single_node(self):
        check_result = RETURN_PASS
        check_cnt = 3
        while check_cnt > 0:
            # 判断水位是否超过90，若有一个VC水位超过90，则继续检查；若都不超过，则返回pass
            check_water_mark_result = self.check_water_mark(self)
            if check_water_mark_result == RETURN_PASS:
                return RETURN_PASS

            # 获取modId为301的实例
            get_inst_id_result = self.get_mds_inst_id(self)[0]
            if get_inst_id_result == RETURN_PASS:
                return RETURN_PASS

            mds_inst_id_s = self.get_mds_inst_id(self)[1]
            for inst_id in mds_inst_id_s:
                check_evict_leak_result = self.check_evict_leak(self, inst_id)
                if check_evict_leak_result == RETURN_NO_PASS:
                    check_result = RETURN_NO_PASS
                    break
            if check_result == RETURN_NO_PASS:
                check_cnt -= 1
                time.sleep(1)
                continue
            return RETURN_PASS
        return RETURN_NO_PASS

    @staticmethod
    def check_water_mark(self):
        """
        检查步骤 1 ：判断水位是否超过90，若有一个VC水位超过90，则继续检查；若都不超过，则返回pass
        """
        rocache_info_s = self.dsl("exec_diagnose 'rocache show ob' | vertical_parser")
        for rocache_info in rocache_info_s:
            result = self.check_water_mark_info(rocache_info)
            if result == RETURN_CONTINUE:
                return RETURN_CONTINUE
        return RETURN_PASS

    @staticmethod
    def check_water_mark_info(rocache_info):
        for key in rocache_info:
            if 'water mark' in key:
                if int(rocache_info[key]) > WATER_MARK_THRESHOLD:
                    return RETURN_CONTINUE
        return RETURN_PASS


    @staticmethod
    def get_mds_inst_id(self):
        """
        检查步骤 2 ：获取mds的实例信息
        """
        mds_inst_id_s = []
        inst_info_s = self.dsl("exec_diagnose 'rocache show inst all all' | horizontal_parser")
        for inst_info in inst_info_s:
            if int(inst_info['modId']) == MDS_MODULE_ID:
                mds_inst_id_s.append(inst_info['Id'])

        if not mds_inst_id_s:
            return RETURN_PASS, ""

        return RETURN_CONTINUE, mds_inst_id_s

    @staticmethod
    def check_evict_leak(self, inst_id):
        """
        检查步骤 3 ：检查实例淘汰优先级5的泄漏情况
        """
        # 按行获取evict对象的回显
        evict_info_s = self.dsl("exec_diagnose 'rocache show ctrevictobj -i {} -t 1'"
                                " | splitlines".format(inst_id))
        for i, evict_info in enumerate(evict_info_s):
            result = self.check_thread_evict_info(i, evict_info, evict_info_s, self)
            if result == RETURN_NO_PASS:
                return RETURN_NO_PASS
        return RETURN_PASS

    @staticmethod
    def check_thread_evict_info(i, evict_info, evict_info_s, self):
        if 'threadId' in evict_info:
            thread_id = int(re.findall(r"\d+\.?\d*", evict_info.split()[0])[0])
            need_evict_num = int(re.findall(r"\d+\.?\d*", evict_info.split()[1])[0])
            if need_evict_num != 0:
                result = self.check_priority_5_info(i, need_evict_num, evict_info_s)
                if result == RETURN_NO_PASS:
                    return RETURN_NO_PASS
        return RETURN_PASS

    @staticmethod
    def check_priority_5_info(i, need_evict_num, evict_info_s):
        count = 0
        while count <= MAX_LOOP_CNT:
            count += 1
            i += 1
            next_evict_info = evict_info_s[i]
            evict_priority_s = next_evict_info.split()
            if not evict_priority_s:
                break
            evict_priority = evict_priority_s[0]
            if evict_priority == '5':
                priority_5_need_evict_num = int(next_evict_info.split()[1])
                priority_5_weight = int(next_evict_info.split()[-1])
                if priority_5_weight > 0 and \
                        need_evict_num / 100 * priority_5_weight < priority_5_need_evict_num:
                    return RETURN_NO_PASS
                break
        return RETURN_PASS
