# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.


from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version


class CheckItem(BaseCheckItem):

    def is_high_enclosure(self):
        enclosures = self.dsl("exec_developer 'show enclosure' | horizontal_parser")
        return bool([enclosure for enclosure in enclosures if "4-controller" in enclosure.get("Type")])

    def has_sas_card(self):
        interfaces = self.dsl("exec_developer 'show interface_module' | horizontal_parser")
        return bool([interface for interface in interfaces if "SAS" in interface.get("Model")])

    def happened_ses_offline(self):
        event_info = self.dsl("exec_developer 'show event level=major object_type=209 |filterRow column=ID "
                              "predict=equal_to value=0xF00D1003D' | horizontal_parser")
        for event in event_info:
            sequence = event.get("Sequence")
            event_detail_info = self.dsl("exec_developer 'show event sequence={}'".format(sequence))
            if "4000D147" in event_detail_info:
                return True
        return False

    def show_alarm_tip(self, version):
        from com.huawei.ism.ui.swing.dialog import DialogUtils
        from com.huawei.ism.tool.bizpack.wizardparse.context import BizPackageContext
        from com.huawei.ism.ui.swing.dialog import WarningDialog
        parent = BizPackageContext.getInstance().getWizard()
        warning_dialog = DialogUtils.showWarnConfirmDialog(parent, self.get_msg("item.suggestion"),
                                                           self.get_msg("check.not.pass.confirm"))
        if warning_dialog.getValue() != WarningDialog.YES_OPTION:
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass", version)
        return CheckStatus.PASS, ""

    def execute(self):
        # 先检查是否是高端带SAS共享卡控制器
        if not self.is_high_enclosure():
            return CheckStatus.PASS, ""
        if not self.has_sas_card():
            return CheckStatus.PASS, ""

        # 6.1.5：SPH12或其之后补丁
        # 1.2.0：SPH12或其之后补丁
        # 由于设置执行环境为6.1.5和1.2.0，因此只需要检查补丁版本大于等于SPH12即可
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        if compare_patch_version(patch_version, "SPH12") >= 0:
            return CheckStatus.PASS, ""

        # 检查SAS共享卡是否发生ses offline
        if self.happened_ses_offline() is True:
            version = version_info.get("base_version").get("Current Version")
            if patch_version != "--":
                version = version + "." + patch_version
            return self.show_alarm_tip(version)

        return CheckStatus.PASS, ""
