# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

from datetime import datetime, timedelta
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus

# 盘编码白名单
CODE_WHITE_LIST = {
    '03032YHR': '28', '03033AUN': '67', '03033CQQ': '72', '03033FKM': '73', '03034AUT': '28',
    '03034AUU': '28', '03033FJB': '72', '03033FKF': '72', '03032WFN': '28', '03033BCW': '28',
    '03033FWL': '67', '03032WFM': '28', '03033FBL': '72', '03033GTB': '72', '03033JPV': '28',
    '03033JTC': '73', '03033QGV': '72', '03033QGW': '73', '03033QGY': '28', '03032VUA': '28',
    '03033FWK': '67', '03032WGK': '14', '03033AAX': '14', '03032YRE': '28', '03033BPF': '28',
    '03033GJS': '72', '03033BDA': '28', '03033FKJ': '73', '03033RXK': '73', '03033GSK': '73',
    '03033QRJ': '73', '03032WET': '28', '03033AXP': '28', '03032VUL': '28', '03032WEJ': '28',
    '03032WEP': '28', '03032YWH': '67', '03033AUP': '67', '03033CQU': '72', '03033CQV': '72',
    '03033GDP': '72', '03032WFG': '14', '03032VUM': '14', '03032WEK': '14', '03032WEQ': '14',
    '03032WFB': '14', '03033CWY': '72', '03033LBA': '37', '03032WFA': '28', '03032WFF': '28',
    '03032YHV': '28', '03033FJH': '73', '03033FKH': '73', '03033FJK': '73', '03033FKL': '73',
    '03033FKN': '73', '03033BGA': '28', '03033FAM': '28', '03033BGB': '14', '03033FAW': '14'
}

# 盘固件白名单
FW_WHITE_LIST = (
    '1009', '1010', '1011', '1012', '1025', '1026', '1027', '1030', '1031', '1033', '1035',
    '1036', '1037', '1038', '2150', '2152', '2153', '2155', '2156', '2158', '2252', '3220',
    '3221', '3222', '3230', '3231', '3232', '3233', '3234', '3235', '3241', '3242', '3243',
    '3245', '3246', '3247', '3248', '3252', '3253', '3255', '1022', '2130', '2131', '2151',
    '3216'
)

# 寿命剩余小于1年，不通过，反馈最小剩余时间
WARNING_NOT_PASS_THRE = 365
# 寿命剩余小于3年，不通过
NOT_PASS_NOT_PASS_THRE = 365 * 3
# 寿命剩余时间大于3年，磨损小于1%
LIFE_REMAIN_FIVE_YEAR = 365 * 5
#最大寿命
LIFE_REMAIN_ENOUGH = 365 * 10


class CheckItem(BaseCheckItem):

    def execute(self):
        # 执行指定命令show disk general
        disk_info_list = self.dsl("exec_developer 'show disk general|filterColumn include columnList=ID,Type,Model,"
                                  "Firmware\\sVersion,Serial\\sNumber,Run\\sTime(Day),Degree\\sof\\sWear(%)'| "
                                  "horizontal_parser")
        if not disk_info_list:
            return CheckStatus.PASS, ""

        sugg_info_lst = []
        warning_lst = []
        no_pass_lst = []
        mini_life = WARNING_NOT_PASS_THRE

        for disk in disk_info_list:
            if not self.check_is_risk_hssd(disk):
                continue

            # 获取当前磨损，上电时间，编码的具体数值，不用判断数据正确性了
            pon, wl, code = self.get_calc_param(disk)
            # 计算盘磨损剩余时长
            remain_time = self.calc_remain_time(pon, CODE_WHITE_LIST.get(code), wl)

            if self.calc_remaintime_achieved_2030(remain_time) or remain_time > LIFE_REMAIN_FIVE_YEAR:
                continue

            # 风险盘根据剩余磨损时长，分类提示
            if remain_time <= WARNING_NOT_PASS_THRE:
                if mini_life > remain_time:
                    mini_life = int(remain_time)
                no_pass_lst.append(disk.get("ID"))
            elif remain_time <= NOT_PASS_NOT_PASS_THRE:
                warning_lst.append(disk.get("ID"))
            else:
                sugg_info_lst.append(disk.get("ID"))

        rlt = self.get_output_info(no_pass_lst, warning_lst, sugg_info_lst, mini_life)

        # 剩余时间小于3年，都是not pass; 大于3年，就提示WARNING; 其他就提示通过
        if no_pass_lst or warning_lst:
            return CheckStatus.NOT_PASS, rlt
        elif sugg_info_lst:
            return CheckStatus.WARNING, rlt
        else:
            return CheckStatus.PASS, ""

    def check_is_risk_hssd(self, disk):
        """
        执行方法：判断合理性 ：代次，编码白名单，固件白名单，上电时长，磨损参数，是否合理可用
        """
        judge_ret = [
            not disk.get("Model").startswith('HSSD-D6'),
            self.get_hssd_code(disk) not in CODE_WHITE_LIST,
            disk.get("Firmware Version") not in FW_WHITE_LIST,
            self.get_param_valid(disk.get("Run Time(Day)")) == -1,
            self.get_param_valid(disk.get("Degree of Wear(%)")) == -1
        ]
        if any(judge_ret):
            return False

        return True

    def get_param_valid(self, pstr):
        """
        执行方法：判断参数能否转换成float类型
        返回值：-1：异常数据不能转换； 其他：成功
        """
        try:
            return float(pstr)
        except ValueError:
            # 异常值反馈-1，因为寿命磨损可能为0，区分一下
            return -1

    def get_calc_param(self, disk):
        """
        执行方法：获取计算的参数，上电时长，磨损，和编码；放在execute导致方法太大
        """
        pon = self.get_param_valid(disk.get("Run Time(Day)"))
        wl = self.get_param_valid(disk.get("Degree of Wear(%)"))
        code = self.get_hssd_code(disk)
        return pon, wl, code

    def get_output_info(self, no_pass_lst, warning_lst, sugg_info_lst, mini_life):
        """
        执行方法： 组装输出的具体提示信息
        """
        rlt = []
        if no_pass_lst:
            rlt.append(self.get_msg("check.not.pass", ", ".join(no_pass_lst), mini_life))
        if warning_lst:
            rlt.append(self.get_msg("check.warning", ", ".join(warning_lst)))
        if sugg_info_lst:
            rlt.append(self.get_msg("check.suggestion", ", ".join(sugg_info_lst)))

        return "\n".join(rlt)

    def get_hssd_code(self, disk):
        """
        执行方法： 通过sn获取code，21打头，取第3到第8位，03打头，用03+第1到第6位
        """
        sn = disk.get("Serial Number")
        if sn.startswith("21") and len(sn) == 20:
            return sn[2:10]
        elif sn.startswith("03") and len(sn) == 16:
            return "03" + sn[0:6]
        return ""

    def calc_remain_time(self, poh, lmt, used):
        """
        执行方法：计算HSSD的剩余时间
        计算公式： 剩余时长 = 当前上电时长（day） / 磨损 * 磨损门限 - 当前使用时长
        """
        if float(used) < 1:
            return LIFE_REMAIN_ENOUGH

        remain_time = float(poh) / float(used) * float(lmt) - float(poh)
        if remain_time < 0:
            return 0
        if remain_time > LIFE_REMAIN_ENOUGH:
            return LIFE_REMAIN_ENOUGH
        return remain_time

    def calc_remaintime_achieved_2030(self, remaining_time):
        """
        执行方法：计算剩余时间是否能够达到2030.1.1
        """
        # 当前日期 + 剩余时间
        today = datetime.now(tz=None)
        calc_time = today + timedelta(days=remaining_time)
        # 目标日期 2030.1.1
        target = datetime(2030, 1, 1)
        return calc_time > target
