# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import re
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg

# eam showevent最多能打印32K字节
FRAME_EVENT_MAX_PRINTF_BYTE = 32 * 1024
RUNNING_STATUS = "Online"
INT_DEFAULT_VALUE = -1
FRAME_ID_ENG_BIT = 24
DAE_X00_ID = 1
DAE_X80_ID = 2


class Frame:
    wwn = ""
    frame_name = ""
    group_id = INT_DEFAULT_VALUE
    frame_id = INT_DEFAULT_VALUE


class FrameEvent:
    wwn = ""
    node_id = INT_DEFAULT_VALUE
    is_first = ""


class FrameEventMap:
    event_list = []
    is_reach_limit = False


class VaultCheckStatus:
    PASS = "PASS"
    LINK_ERROR = "LINK_ERROR"
    RECORD_ERROR = "RECORD_ERROR"
    STATUS_ERROR = "STATUS_ERROR"
    WARNING = "WARNING"


class NodeResultMap:
    node_id = INT_DEFAULT_VALUE
    group_id = INT_DEFAULT_VALUE
    abnormal_frame = None
    check_status = VaultCheckStatus.PASS


class CheckItem(BaseCheckItem):


    @staticmethod
    def is_need_collect_event(wwn, vault_frame_list):
        """
        是否需要收集框事件：只有保险框才收集
        """
        for frame in vault_frame_list:
            if frame.wwn == wwn:
                return True
        return False

    @staticmethod
    def merge_diag_vault_frames(old_frames, new_frame_set):
        """
        合并保险框数据
        """
        for frame in new_frame_set:
            if frame.frame_name not in old_frames:
                old_frames[frame.frame_name] = frame

    def execute(self):
        self.is_6u4c_sas_frame = False
        # 节点号与分组的映射表
        self.node_group_map = {}
        self.logger.info("Start check --------------------")
        # 若不是超级管理员用户，则检查结果为检查未检查
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")
        self.init_node_group_map()
        self.init_6u4c_sas_frame_flag()
        status, err_msg = self.check_x86_device()
        return status, err_msg

    def is_vault_frame(self, frame_name):
        """
        判断是否是保险框，DAEx00一定是保险框，6U4C sas小卡环境DAEx80也是保险框
        """
        if re.search(r"DAE\d00", frame_name) is not None:
            return True
        if self.is_6u4c_sas_frame and re.search(r"DAE\d80", frame_name) is not None:
            return True
        return False

    def get_id_by_frame_name(self, frame_name):
        """
        根据保险框的框名称获取所在的引擎id和分组id
        """
        eng_id = int(frame_name[3])
        group_id = eng_id
        frame_id = (eng_id << FRAME_ID_ENG_BIT) | DAE_X00_ID
        if self.is_6u4c_sas_frame:
            if re.search(r"DAE\d00", frame_name) is not None:
                group_id = eng_id * 2
            if re.search(r"DAE\d80", frame_name) is not None:
                group_id = eng_id * 2 + 1
                frame_id = (eng_id << FRAME_ID_ENG_BIT) | DAE_X80_ID
        return group_id, frame_id

    def init_node_group_map(self):
        node_infos = self.dsl("exec_diagnose 'sys showcls' | horizontal_parser")
        for node_info in node_infos:
            # 获取节点对应的group id
            node_id = int(node_info.get("id"))
            group_id = int(node_info.get("group"))
            self.node_group_map[node_id] = group_id
        self.logger.info("Init node group map {} finish.".format(self.node_group_map))

    def init_6u4c_sas_frame_flag(self):
        """
        判断是否是6U4C SAS小卡环境带C、D控，这种环境的保险框为DAEX80
        """
        model = self.context.dev_node.model
        if re.search(r"6800.*V3|Dorado18000.*V3|6900.*V3", model) is None:
            self.is_6u4c_sas_frame = False
        else:
            self.is_6u4c_sas_frame = True
        self.logger.info("The model is {}, is 6u4c sas env {}.".format(model, self.is_6u4c_sas_frame))

    def query_cli_vault_frames(self):
        """
        cli查询保险框
        """
        vault_frames = {}
        frames = self.dsl("exec_developer 'show enclosure' | horizontal_parser")
        for frame in frames:
            frame_name = frame.get("ID")
            running_status = frame.get("Running Status")
            if self.is_vault_frame(frame_name):
                vault_frames[frame_name] = running_status
        self.logger.info("Vault frame is {}.".format(vault_frames))
        return vault_frames

    def check_vault_frames_status(self, vault_frames):
        vault_abnormal_set = set()
        for name, status in vault_frames.items():
            if status != RUNNING_STATUS:
                vault_abnormal_set.add(name)
                self.logger.error("Vault frame {} status abnormal ,is {}.".format(name, status))
        return vault_abnormal_set

    def check_vault_frames_number(self, cli_vault_frames, diag_vault_frames):
        """
        检查保险框数CLI与diagnose是否一致，diagnose查询的框记录上限是32K
        cli_vault_frames：字典，diagnose_vault_frames：字典
        检查失败返回vault_frame_abnormal_list(查询失败的框)
        检查成功返回vault_frame_list(单个查询到的框)
        """
        vault_frame_set = set()
        vault_frame_abnormal_set = set()
        if len(cli_vault_frames) == len(diag_vault_frames):
            return VaultCheckStatus.PASS, vault_frame_set
        for frame_name in cli_vault_frames.keys():
            if frame_name not in diag_vault_frames:
                _, frame_id = self.get_id_by_frame_name(frame_name)
                cmd = "exec_diagnose 'eam showframe {}'| splitlines".format(frame_id)
                frame_dict = self.query_diagnose_frame_records(cmd)
                if frame_name not in frame_dict:
                    vault_frame_abnormal_set.add(frame_name)
                    continue
                vault_frame_set.add(frame_dict[frame_name])
        if vault_frame_abnormal_set:
            self.logger.error("Get single vault frame {} failed.".format(vault_frame_abnormal_set))
            return VaultCheckStatus.RECORD_ERROR, vault_frame_abnormal_set
        return VaultCheckStatus.PASS, vault_frame_set

    def query_diagnose_frame_records(self, cmd):
        """
        查询框记录，这里只需要记录保险框信息
        """
        result_list = self.dsl(cmd)
        wwn_tmp = ""
        vault_frame_dict = {}
        for line in result_list:
            line_split = str(line).split()
            if len(line_split) <= 2:
                continue
            if re.search(r"^wwn\s+.*", line) is not None:
                wwn_tmp = line_split[2]
            if re.search(r"^frameName\s+.*", line) is not None:
                frame_name = line_split[2]
                # 只统计保险框信息
                if not self.is_vault_frame(frame_name):
                    continue
                frame = Frame()
                frame.wwn = wwn_tmp
                frame.frame_name = frame_name
                frame.group_id, frame.frame_id = self.get_id_by_frame_name(frame_name)
                vault_frame_dict[frame_name] = frame
                self.logger.info("Vault frame name {},wwn {},group {},frame_id {}."
                                 .format(frame_name, wwn_tmp, frame.group_id, frame.frame_id))
        return vault_frame_dict

    def query_frame_event_list(self, vault_frame_list):
        """
        diagnose下查询框事件，node id表示事件上报的节点，is first表示是否接的保险框的位置
        """
        result = self.dsl("exec_diagnose 'eam showevent'")
        result = str(result)
        result_map = FrameEventMap()
        node_id = INT_DEFAULT_VALUE
        wwn_tmp = ""
        event_list = []
        result_list = result.split("\n")
        for line in result_list:
            line_split = line.split()
            if re.search(r"^node\s+Id\s+.*", line) is not None and len(line_split) > 3:
                node_id = int(line_split[3])
            if re.search(r"^wwn\s+.*", line) is not None and len(line_split) > 2:
                wwn_tmp = line_split[2]
            if re.search(r"^Is\s+first\s+.*", line) is not None and len(line_split) > 3:
                is_first = line_split[3]
                # 只统计保险框事件
                if not self.is_need_collect_event(wwn_tmp, vault_frame_list):
                    continue
                event = FrameEvent()
                event.node_id = node_id
                event.is_first = is_first
                event.wwn = wwn_tmp
                event_list.append(event)
        result_map.event_list = event_list

        # 判断框事件是否已经超过上限，eam showevent最多能查32K数据
        if len(result) >= FRAME_EVENT_MAX_PRINTF_BYTE:
            result_map.is_reach_limit = True
        else:
            result_map.is_reach_limit = False
        return result_map

    def check_event_by_record(self, frame, event_map, local_group):
        event_count = 0
        for event in event_map.event_list:
            self.logger.info("Event wwn {}, node {}, frame wwn {}, group {}, group map {}."
                             .format(event.wwn, event.node_id, frame.wwn, frame.group_id, self.node_group_map))
            if (frame.wwn != event.wwn) or (local_group != self.node_group_map.get(event.node_id)):
                continue
            if event.is_first != "1":
                self.logger.error("Vault frame {} group id {} is link error.".format(frame.frame_name, local_group))
                return VaultCheckStatus.LINK_ERROR
            if event.is_first == "1":
                event_count += 1
        if event_count == 0:
            # 如果没有对应的事件，则需要判断是否超过32K
            self.logger.error("Vault frame {} group id {} event lost.".format(frame.frame_name, local_group))
            if event_map.is_reach_limit:
                return VaultCheckStatus.WARNING
            return VaultCheckStatus.LINK_ERROR
        return VaultCheckStatus.PASS

    def get_local_node_id(self):
        """
        获取当前节点的node id
        """
        local_id = INT_DEFAULT_VALUE
        dev_infos = self.dsl("exec_diagnose 'sys showcls' | vertical_parser")
        if not dev_infos:
            self.logger.info("Execute cmd sys showcls error.")
            return local_id
        dev_info = dev_infos[0]
        local_id = int(dev_info.get("local node id"))
        return local_id

    def deal_node_event(self, vault_frame_list):
        """
        处理每个节点的保险框事件
        """
        node_result_map = NodeResultMap()
        local_id = self.get_local_node_id()
        local_group = self.node_group_map.get(local_id)
        if (local_id == INT_DEFAULT_VALUE) or (local_group == INT_DEFAULT_VALUE):
            self.logger.error("Get node info failed.")
            return node_result_map

        node_result_map.node_id = local_id
        node_result_map.group_id = local_group
        node_result_map.check_status = VaultCheckStatus.PASS
        event_map = self.query_frame_event_list(vault_frame_list)
        for frame in vault_frame_list:
            self.logger.info("Frame group {}, name {}.".format(frame.group_id, frame.frame_name))
            # 只判断同组的框记录
            if frame.group_id != local_group:
                continue
            # 每个保险框在分组内只有存在1个
            check_status = self.check_event_by_record(frame, event_map, local_group)
            if check_status == VaultCheckStatus.PASS:
                break
            else:
                node_result_map.check_status = check_status
                node_result_map.abnormal_frame = frame
                break
        return node_result_map

    def deal_node_check_result(self, nodes_result):
        """
        1、每个分组内只有1个保险框，分组内存在检查通过的节点，则不会触发告警
        2、存在检查错误的节点，则优先返回检查错误信息
        3、只有分组内所有节点检查都是告警，才触发告警
        """
        group_has_warning = {}
        abnormal_vault_set = set()
        has_error = False
        for key, node_result in nodes_result.items():
            if node_result.node_id == INT_DEFAULT_VALUE:
                self.logger.error("Node {} query cmd failed.".format(key))
                continue
            group_id = node_result.group_id
            if node_result.check_status == VaultCheckStatus.PASS:
                group_has_warning[group_id] = False
                self.logger.info("Node {} check pass.".format(node_result.node_id))
                continue
            self.logger.error("Node {} check abnormal with frame {}."
                              .format(node_result.node_id, node_result.abnormal_frame.frame_name))
            if node_result.check_status != VaultCheckStatus.WARNING:
                has_error = True
                abnormal_vault_set.add(node_result.abnormal_frame.frame_name)
            if node_result.check_status == VaultCheckStatus.WARNING:
                # 如果存在检查错误，则无需检查告警
                if has_error:
                    continue
                # 如果分组内key值存在，但不存在告警，则说明有节点通过
                if (group_id in group_has_warning) and (not group_has_warning.get(group_id)):
                    continue
                # key不存在，或已存在告警，则置标志位True，记录异常框信息
                if group_id not in group_has_warning or group_has_warning.get(group_id):
                    group_has_warning[group_id] = True
                    abnormal_vault_set.add(node_result.abnormal_frame.frame_name)
        return has_error, abnormal_vault_set

    def check_x86_device(self):
        """
        1、CLI查询到的保险框状态异常，则检查不通过
        2、如果diagnose查询到的单个框记录失败，则检查不通过
        3、如果节点存在检查组网错误，则检查不通过
        4、如果分组内的所有节点都检查出告警，则检查告警
        """
        cli_vault_frames = self.query_cli_vault_frames()
        # 1、检查CLI查询到的保险框状态是否异常，异常则检查不通过
        abnormal_vault_set = self.check_vault_frames_status(cli_vault_frames)
        if abnormal_vault_set:
            result_list = list(abnormal_vault_set)
            result_list.sort()
            return CheckStatus.NOT_PASS, self.get_msg("check.status.abnormal", ",".join(result_list))

        cmd = "exec_diagnose 'eam showframe'| splitlines"
        diag_vault_frames = self.query_diagnose_frame_records(cmd)
        # 2、检查框记录是否缺失，大规格场景下，框记录可能打印不全，如果有缺失则查单个信息补全
        status, vault_set = self.check_vault_frames_number(cli_vault_frames, diag_vault_frames)
        if status != VaultCheckStatus.PASS:
            vault_list = list(vault_set)
            vault_list.sort()
            return CheckStatus.NOT_PASS, self.get_msg("check.record.error", ",".join(vault_list))
        if vault_set:
            self.merge_diag_vault_frames(diag_vault_frames, vault_set)

        # 3、检查错误的优先级比告警优先级更高，只要存在组网错误，则无需判断告警
        nodes_result = self.dsl("exec_on_all {}", self.deal_node_event, list(diag_vault_frames.values()))
        has_error, abnormal_vault_set = self.deal_node_check_result(nodes_result)
        result_list = list(abnormal_vault_set)
        result_list.sort()
        if has_error:
            self.logger.error("Check not pass with frame {}.".format(result_list))
            return CheckStatus.NOT_PASS, self.get_msg("check.link.error", ",".join(result_list))
        # 4、如果分组内的所有节点都检查出告警，则检查告警
        if abnormal_vault_set:
            self.logger.error("Check warning with frame {}.".format(result_list))
            return CheckStatus.WARNING, self.get_msg("check.warning", ",".join(result_list))
        return CheckStatus.PASS, ""
