# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version

VER_DICT = {'6.0.1': 'SPH35', '6.1.0': 'SPH27', '6.1.2': 'SPH30', '6.1.3': 'SPH10', '6.1.5': 'SPH8',
            'V500R007C60SPC100': 'V500R007C60SPH111',
            'V500R007C60SPC300': 'V500R007C60SPH330',
            'V500R007C70SPC200': 'V500R007C70SPH210',
            'V500R007C71SPC100': 'V500R007C71SPH130',
            'V500R007C73SPC100': 'V500R007C73SPH110'}

PRODUCT_MODEL_H = [
    "OceanStor Dorado 8000 V6", "OceanStor Dorado 18000 V6", "OceanStor Dorado 6800 V6",
    "OceanStor Dorado 18500 V6", "OceanStor Dorado 18800 V6", "OceanStor Dorado 18800K V6",
    "OceanStor 6810", "OceanStor 18510", "OceanStor 18500K", "OceanStor 18810",
    "6800 V5", "6800F V5", "6800K V5", "6810 V5", "6810F V5", "18500 V5", "18500F V5", "18500K V5",
    "18510 V5", "18510F V5", "18800 V5", "18800F V5", "18800K V5", "18810 V5", "18810F V5",
]


class CheckItem(BaseCheckItem):
    def execute(self):
        self.logger.info("power_psu_upgrade_check start!")
        version_info = get_version_info(self.dsl)
        software_version = version_info.get("base_version").get("Current Version")
        hotpatch_version = version_info.get("patch_version").get("Current Version")

        if software_version not in VER_DICT:  # 版本
            return CheckStatus.PASS, ""
        result = compare_patch_version(hotpatch_version, VER_DICT.get(software_version))  # 补丁
        if result < 0:
            return CheckStatus.PASS, ""
        if self.check_psu_target(software_version):
            return CheckStatus.PASS, ""

        # 控制器0节点查询电源升级状态
        results = self.dsl("exec_on_all {}", self.check_psu_upgrade, target_ctrl={"0": "0"})
        if not results.values():
            self.logger.info('exec_on_all results is None.')
            return CheckStatus.NOT_PASS, self.get_err_msg("check.message.exception",
                                                          "item.message.exception.suggestion")
        self.logger.info('exec_on_all results is not None.')
        for result in results.values():
            if result:
                return CheckStatus.PASS, ""
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")

    def check_psu_target(self, software_ver):
        product_model = self.context.dev_node.model
        exec_cmd = "exec_cli 'show power_supply|filterColumn include columnList=ID,Version,Model,Serial\sNumber'"
        result_msg = self.dsl(exec_cmd)
        psu_list = self.dsl("echo {} | horizontal_parser", result_msg)
        for psu_info in psu_list:
            psu_id = psu_info.get("ID", "")
            psu_model = psu_info.get("Model", "")
            psu_verdc = psu_info.get("Version", "")
            if "CTE" in psu_id and product_model in PRODUCT_MODEL_H: # 高端控制框PSU
                continue
            # 存在需要升级psu
            if psu_model == "PAC2000S12-BG":
                if software_ver in ['6.0.1', 'V500R007C60SPC100', 'V500R007C60SPC300']:
                    return False
                if psu_verdc != "" and psu_verdc.split(".")[0] in ['111', '113', '115', '119']:
                    return False
        return True  # 不存在需要升级psu

    def check_psu_upgrade(self):
        is_exist_permit = self.check_file_exist("power_upgrade_permit")
        is_exist_prohibit = self.check_file_exist("power_upgrade_prohibit")
        if is_exist_permit:
            return is_exist_prohibit
        return True

    def check_file_exist(self, filename):
        conf_result = self.dsl("exec_mini_system 'ls /startup_disk/conf/conf_local/'")
        tmp_result = self.dsl("exec_mini_system 'ls /tmp/'")
        if filename in conf_result or filename in tmp_result:
            return True
        return False
