# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.
import re
import ast

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.checkitem.common.power_replace_common import self_exit, check_subrack_type


class CheckItem(BaseCheckItem):
    def execute(self):
        # 对于风扇模块，如果后续也要做该检查项，注意风扇模块可以多选，并且可能object_id参数中有空格
        selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
        # 获取name值
        object_name = selected_fru.get("name", "")
        self.logger.info(object_name)
        script_name = 'power_replace_before_entry_conditions_check'
        try:
            if not check_subrack_type(self, object_name, script_name):
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass.subract", "item.suggestion.subract")
            self.logger.info(script_name + " start!")
            # 控制器0节点查询电源升级状态
            results = self.power_dev_mod(object_name)
            if results:
                return CheckStatus.PASS, ""
            else:
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
        finally:
            self_exit(self, script_name)

    # 当前设备形态检查
    def power_dev_mod(self, object_name):
        dev_mod_status = self.dsl("exec_mini_system 'eam.sh showenclosure'")
        # 获取frameType、enclModel字段信息
        frame_type_info = re.search(r'frameType\s*=\s*\w+', dev_mod_status).group()
        encl_model_info = re.search(r'enclModel\s*=\s*\w+', dev_mod_status).group()
        # 获取frameType、enclModel具体值
        index_f = frame_type_info.index("= ") + 2
        index_e = encl_model_info.index("= ") + 2
        mid_model = ['0x71', '0x72', '0x73', '0x7b', '0x7c', '0x7d', '0x7e']
        low_model = ['0x74', '0x75', '0x76', '0x77', '0x78', '0x7a']
        if frame_type_info[index_f:] == 'CtrlEncl' and (encl_model_info[index_e:] in mid_model):
            return True
        if frame_type_info[index_f:] == 'CtrlEncl' and (encl_model_info[index_e:] in low_model):
            return self.check_entry_conditions(object_name)
        if frame_type_info[index_f:] == 'IpEncl' and (encl_model_info[index_e:] in mid_model or \
                                                      encl_model_info[index_e:] in low_model):
            return self.check_entry_conditions(object_name)
        else:
            return False

    # 入口条件检查
    def check_entry_conditions(self, object_name):
        # 电源功率查询
        power_psu0_pow_info = self.dsl("exec_mini_system 'eam.sh psu_upgrade_avgpower_check 1000 0'")
        power_psu1_pow_info = self.dsl("exec_mini_system 'eam.sh psu_upgrade_avgpower_check 1001 0'")
        # 获取回显中"power："开始的信息
        power_psu0_pow = re.search(r'power:\d+', power_psu0_pow_info).group()
        power_psu1_pow = re.search(r'power:\d+', power_psu1_pow_info).group()
        # 获取具体功率数值
        index_psu0 = power_psu0_pow.index("power:") + 6
        index_psu1 = power_psu1_pow.index("power:") + 6
        psu0_pow = int(power_psu0_pow[index_psu0:])
        psu1_pow = int(power_psu1_pow[index_psu1:])
        power_total = int(psu0_pow) + int(psu1_pow)
        # 跳转0节点
        results = self.dsl("exec_on_all {}", self.check_ctrl_fan_speed, object_name, power_total, \
                            target_ctrl={"0": "0"})
        if not results.values():
            return False
        for result in results.values():
            if result:
                return True
            return False

    def check_ctrl_fan_speed(self, object_name, power_total):
        # 跳转0节点后执行判断总功率大于600W，避免return后退出异常
        if power_total >= 600:
            return False
        fan_info = self.dsl("exec_cli 'show fan'")
        lines = list(map(lambda x: x.strip(), fan_info.split("\n")))

        # 风扇状态不是状态Normal或转速不是Low，认为不符合
        for line in lines:
            if line.startswith(object_name):
                if ' Normal ' in line and ' Low' in line:
                    continue
                else:
                    return False
        return True

