# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import os
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version

PATCH_VERSION_DICT = {
    '6.0.0': '--', '6.0.1': 'SPH35', '6.1.0': 'SPH27', '6.1.2': 'SPH30', '6.1.3': 'SPH10', '6.1.5': 'SPH8',
    'V500R007C60': '--', 'V500R007C60SPC100': 'V500R007C60SPH111', 'V500R007C60SPC300': 'V500R007C60SPH330',
    'V500R007C70': '--', 'V500R007C70SPC100': '--', 'V500R007C70SPC200': 'V500R007C70SPH210',
    'V500R007C71': '--', 'V500R007C71SPC100': 'V500R007C71SPH130',
    'V500R007C73': '--', 'V500R007C73SPC100': 'V500R007C73SPH110',
}  # '6.0.1', 'V500R007C60SPC100', 'V500R007C60SPC300'版本做内置检查项，这里不再检查

PRODUCT_MODEL_H = [
    "OceanStor Dorado 8000 V6", "OceanStor Dorado 18000 V6", "OceanStor Dorado 6800 V6",
    "OceanStor Dorado 18500 V6", "OceanStor Dorado 18800 V6", "OceanStor Dorado 18800K V6",
    "OceanStor 6810", "OceanStor 18510", "OceanStor 18500K", "OceanStor 18810",
    "6800 V5", "6800F V5", "6800K V5", "6810 V5", "6810F V5", "18500 V5", "18500F V5", "18500K V5",
    "18510 V5", "18510F V5", "18800 V5", "18800F V5", "18800K V5", "18810 V5", "18810F V5",
]


class CheckItem(BaseCheckItem):
    def execute(self):
        product_model = self.context.dev_node.model
        version_info = get_version_info(self.dsl)
        software_version = version_info.get("base_version").get("Current Version")
        hotpatch_version = version_info.get("patch_version").get("Current Version")
        self.logger.info("Product Model:{}. ".format(product_model))
        self.logger.info("Software Version:{}, Hotpatch Version:{}.".format(software_version, hotpatch_version))

        exec_cmd = "exec_cli 'show power_supply|filterColumn include columnList=ID,Version,Model,Serial\sNumber'"
        result_msg = self.dsl(exec_cmd)
        psu_sn_list = self.dsl("echo {} | horizontal_parser", result_msg)
        psu_risk_msg = ""
        psu_warn_msg = ""
        for psu_sn in psu_sn_list:
            psu_id = psu_sn.get("ID", "")
            psu_model = psu_sn.get("Model", "")
            psu_version = psu_sn.get("Version", "")
            psu_sn_code = psu_sn.get("Serial Number", "")
            psu_bom, psu_date = psu_sn_code[2:10], psu_sn_code[12:14]
            if "CTE" in psu_id and product_model in PRODUCT_MODEL_H: # 高端控制框PSU
                continue
            psu_msg = "ID:{}, Model:{}, Version:{}, Serial Number: {}\n" \
                      .format(psu_id, psu_model, psu_version, psu_sn_code)
            # 风险电源Model为PAC2000S12-BG, Serial Number包含BOM编码02312HGY/02352TYV
            if not (psu_model == "PAC2000S12-BG" and psu_bom in ['02312HGY', '02352TYV']):
                continue
            if psu_version != "" and psu_version.split(".")[0] in ['111', '113', '115', '119']:
                if psu_date <= "N3":
                    psu_risk_msg += psu_msg
                else:
                    psu_warn_msg += psu_msg

        ret_err_msg = ""
        if psu_risk_msg != "":
            result = compare_hotpatch_version(hotpatch_version, PATCH_VERSION_DICT.get(software_version))
            if result:  # 安装电源升级补丁
                ret_err_msg += self.get_msg("check.not.upgrade")
            else:
                ret_err_msg += self.get_msg("check.not.pass")
            ret_err_msg += psu_risk_msg
        if psu_warn_msg != "":
            ret_err_msg += self.get_msg("check.psu.after.n3")
            ret_err_msg += psu_warn_msg

        if psu_risk_msg != "":
            return CheckStatus.NOT_PASS, ret_err_msg
        elif psu_warn_msg != "":
            return CheckStatus.WARNING, ret_err_msg
        return CheckStatus.PASS, ""


def compare_hotpatch_version(version1, version2):
    if version2 == "--":
        return False
    result = compare_patch_version(version1, version2)
    if result == -1:
        return False
    return True
