# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from expparser import *
import re
import traceback
from psdk.dsl.operator.base_operator import BaseOperator
from psdk.platform.util.echo_parser import get_horizontal_cli_ret, get_vertical_cli_ret


class ParserDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "parser"

    def get(self):
        c = Suppress("|")
        parse_key = (Literal("vertical_parser")
                     | Literal("horizontal_parser")).setResultsName("type")
        pattern = c + parse_key
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        if toks.type == "vertical_parser":
            return get_vertical_cli_ret(self.dsl_context.last_data)
        else:
            return get_horizontal_cli_ret(self.dsl_context.last_data)


class RegexDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "parser"

    def get(self):
        c = Suppress("|")
        not_char = CharsNotIn("'")
        parse_key = Literal("regex").setResultsName("type")
        quota = Suppress("'")
        regex = quota + not_char.setResultsName("regex") + quota
        expression = c + parse_key + regex
        expression.setParseAction(self.base_action)
        return expression

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        data = self.dsl_context.last_data
        if not data:
            return None
        if not isinstance(data, list):
            return self._search(toks.regex, data)

        ret = []
        for item in data:
            sections = self._search(toks.regex, item)
            if sections:
                ret.append(sections)
        return ret

    @staticmethod
    def _search(pattern, text):
        matched = re.search(pattern, text)
        if not matched:
            return ""
        elem = matched.groupdict()
        if not elem:
            elem = matched.groups()
        return elem


class SplitlinesDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "parser"

    def get(self):
        c = Suppress("|")
        parse_key = Literal("splitlines").setResultsName("type")
        pattern = c + parse_key
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        return self.dsl_context.last_data.splitlines()


class SplitDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "parser"

    def get(self):
        c = Suppress("|")
        parse_key = (Literal("split")).setResultsName("type")
        a = CharsNotIn("'")
        b = Suppress("'")
        symbol = b + a.setResultsName("symbol") + b
        pattern = c + parse_key + symbol
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        data = self.dsl_context.last_data
        if not isinstance(data, list):
            return re.split(toks.symbol, data)
        ret = []
        for item in data:
            ret.append(re.split(toks.symbol, item))
        return ret


class GetKeyDsl(BaseOperator):
    def __init__(self, dsl_context, data=None):
        BaseOperator.__init__(self, dsl_context, data)
        self.operator_type = "parser"

    def get(self, alone=True):
        c = Suppress("|")
        parse_key = (Literal("get_key")).setResultsName("type")
        filed_name = Regex(r"[^']+").setResultsName("filed_name")
        pattern = parse_key + Suppress("('") + filed_name + Suppress("')")
        if alone:
            pattern = c + pattern
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        field_name = toks.filed_name
        # 用类型+字段名作为key
        key = toks.type + toks.filed_name
        if isinstance(self.data, list):
            ret = []
            for item in self.data:
                ret.append(item.get(field_name, ''))
            toks[key] = ret
            return ret

        # 将提取到的值存入toks中
        toks[key] = self.data.get(field_name, '')
        return toks[key]


class GetIndexDsl(BaseOperator):
    def __init__(self, dsl_context, data=None):
        BaseOperator.__init__(self, dsl_context, data)
        self.operator_type = "parser"

    def get(self, alone=True):
        c = Suppress("|")
        parse_key = (Literal("get_index")).setResultsName("type")
        index = Word(nums).setResultsName("index")
        pattern = parse_key + Suppress("(") + index + Suppress(")")
        if alone:
            pattern = c + pattern
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        # 用类型+字段名作为key
        index = int(toks.index)
        key = toks.type + toks.index
        toks[key] = self.data[index]
        return self.data[index]


class FilterDsl(BaseOperator):
    """
    过滤操作:从list中筛选出符合条件的数据，没有返回[]
    """
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "parser"

    def get(self):
        c = Suppress("|")
        operate_key = Literal("filter").setResultsName("type")
        express = Combine(OneOrMore(CharsNotIn("|")))\
            .setResultsName("inner_express")
        pattern = c + operate_key + express
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        ret = []
        for one_data in self.data:
            pattern = ExpressDsl(self.dsl_context, one_data).get()
            try:
                pattern.parseString(toks.inner_express)
            except Exception:
                self.logger.info("parse filter error."
                                 + traceback.format_exc())
                continue
            if self.dsl_context.last_data:
                ret.append(one_data)
        return ret


class ExpressDsl(BaseOperator):
    """
    逻辑运算操作
    支持嵌套get运算
    """
    def __init__(self, dsl_context, data=None):
        BaseOperator.__init__(self, dsl_context, data)
        self.operator_type = "parser"

    def get(self):
        operator = (Literal(">")
                    | Literal("==")
                    | Literal("<")
                    | Literal("!=")).setResultsName("operator")
        operator_num = (Word(nums) | Suppress("'") + CharsNotIn("'")
                        + Suppress("'")).setResultsName("operator_num")
        get_key_dsl = \
            Combine(GetKeyDsl(self.dsl_context, self.data).get(False)) \
            + operator + operator_num
        get_index_dsl = \
            GetIndexDsl(self.dsl_context, self.data).get(False) \
            + operator + operator_num
        pattern = get_key_dsl | get_index_dsl
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        express = []
        for item in toks:
            if item in toks:
                express.append(toks[item])
            else:
                express.append(item)
        if len(express) == 3 and \
                (not express[0].isdigit() or not express[2].isdigit()):
            express[0] = "'" + express[0] + "'"
            express[2] = "'" + express[2] + "'"
        return eval(" ".join(express))


class LogicDsl(BaseOperator):
    """
    逻辑运算操作
    支持嵌套get运算
    """
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "parser"

    def get(self):
        c = Suppress("|")
        parse_key = Literal("logic").setResultsName("type")
        express = ExpressDsl(self.dsl_context).get()
        pattern = c + parse_key + express
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        """
        toks既是数组，也是字典
        toks ['logic', 'get_keyfault_page_num', '>', '20']
        filed_name = {str} 'fault_page_num'
        get_keyfault_page_num = {str} '10'
        operator = {str} '>'
        operator_num = {str} '20'
        type = {str} 'get_key'
        """
        self.logger.info("enter {} action ".format(toks.type))
        return self.dsl_context.last_data


class BreakDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "parser"

    def get(self):
        c = Suppress("|")
        ret_key = (Literal("return") | Literal("raise")).setResultsName("type")
        ret_value = Literal("true") | Literal("false")
        condition_key = Literal("if")
        condition_value = Literal("empty")

        pattern = c + ret_key + ret_value + condition_key + condition_value
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        return


class EchoDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.operator_type = "data_source"

    def get(self):
        echo_key = Literal("echo").setResultsName("type")
        pattern = echo_key + CharsNotIn("|")
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        return self.dsl_context.args[0]
