# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import traceback

from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.echo_parser import get_horizontal_cli_ret
from psdk.platform.protocol.cli_service import exec_cli, exec_diagnose, exec_mini_system
from psdk.platform.util.echo_parser import get_vertical_cli_ret
from psdk.platform.protocol.cli_util import isInStorageMode, execute_cmd

"""
执行CLI命令，封装特定的基础业务的接口
方法统一返回格式要求：（status, data, origin_info）
支持静默忽略异常、
可能的错误提示：
执行命令（xx）失败
建议：请检查设备状态和网络连接是否正常后重试，重试失败请联系技术工程师处理。
考虑偏函数的使用
"""


def get_controller_ip_list(cli, lang):
    """获取所有控制器的ip列表

    :param cli: cli
    :param lang: lang
    :return:
    """
    controller_ip_list = []
    cmd = "show upgrade package"
    flag, cli_ret = exec_cli(cli, cmd, lang)
    if not flag:
        return False, controller_ip_list

    hot_patch_begin_index = cli_ret.find("HotPatch Version")
    software_begin_index = cli_ret.find("Software Version")
    if software_begin_index == -1 or hot_patch_begin_index == -1:
        return False, ""
    software_list = get_horizontal_cli_ret(
        cli_ret[software_begin_index:hot_patch_begin_index])
    for software in software_list:
        controller_ip = software.get("IP")
        controller_ip_list.append(controller_ip)

    return True, controller_ip_list


def get_system_version_with_ret(cli, lang):
    cmd = "show upgrade package"
    ok, cli_ret = exec_cli(cli, cmd, lang)
    if not ok:
        return False, "", "", cli_ret

    hot_patch_begin_index = cli_ret.find("HotPatch Version")
    software_begin_index = cli_ret.find("Software Version")
    if software_begin_index == -1 or hot_patch_begin_index == -1:
        return False, "", "", cli_ret
    software_dict = get_horizontal_cli_ret(
        cli_ret[software_begin_index:hot_patch_begin_index])
    software_version = software_dict[0].get('Current Version')

    hot_patch_dict = get_horizontal_cli_ret(cli_ret[hot_patch_begin_index:])
    valid_hotpatch_vers = list(filter(
        lambda hdict: hdict.get('Current Version') != '--' and
        hdict.get('Current Version'), hot_patch_dict))
    if list(valid_hotpatch_vers):
        hot_patch_version = valid_hotpatch_vers[0].get('Current Version')
    else:
        hot_patch_version = ''

    return True, software_version, hot_patch_version, cli_ret


def get_engine_ctrl_topography(cli, lang):
    """
    @summary: fetch all enginId and corrosponding controllerIdlist info:

    :returns
    isOK:
    cliRet:
    erMsg:
    ctlrsTopographyTuple:(currentEngineId,currentCtlrId, {engineID: [ctlrIds]}

    """
    ctlrs_infos = ("", "", {})
    # 先尝试执行小系统下的指令
    is_exe_succ, cli_ret = exec_diagnose(cli, "sys showcls", lang)

    if not is_exe_succ:
        return False, cli_ret, ctlrs_infos

    engine_node_dict = {}
    cur_engine_id = ""
    current_node = ""
    vertical_ret = get_vertical_cli_ret(cli_ret)
    horizontal_ret = get_horizontal_cli_ret(cli_ret)

    current_node = vertical_ret[0].get("local node id")
    for horizontal_info in horizontal_ret:
        engine_id = horizontal_info.get("engine")
        controller_id = horizontal_info.get("id")
        if engine_id not in engine_node_dict:
            engine_node_dict[engine_id] = []
        if current_node == controller_id:
            cur_engine_id = engine_id
        engine_node_dict[engine_id].append(controller_id)

    ctlrs_topography_tuple = (cur_engine_id, current_node, engine_node_dict)

    return True, cli_ret, ctlrs_topography_tuple


def exitHeartbeatCli(cli, lang):
    """
    @summary: Exit from minisystem mode of peer controller to local controller CLI mode.
    @param cli: CLI connection
    @param lang: 语言
    @return:None
    """
    try:
        for _ in range(3):
            _, cli_ret = exec_cli(cli, "exit", lang)

            while "(y/n)" in cli_ret:
                _, cli_ret = exec_cli(cli, "y", lang)

            # 兼容debug包退出到storage模式
            if isInStorageMode(cli_ret):
                _, cli_ret = exec_cli(cli, "exit", lang)
    except Exception:
        traceback.format_exc(None)


def sshToRemoteContr(cli, cmd, psd, lang):
    """
    @summary: 在minisystem模式执行心跳命令到指定的控制器
    """
    all_cli_ret = ""
    err_msg = ""
    flag, cli_ret = exec_mini_system(cli, cmd, lang)
    all_cli_ret += cli_ret
    if not flag:
        return flag, cli_ret, err_msg

    # 首次心跳问题.
    if '(yes/no)?' in cli_ret:
        all_cli_ret += "\n" + cli_ret
        flag, cli_ret = exec_cli(cli, 'yes', lang)

    if "password:" in cli_ret:
        flag, cli_ret = execute_cmd(cli, psd, False)
        all_cli_ret += "\n" + cli_ret
        if not flag:
            return flag, all_cli_ret, err_msg

    if "System Name" not in cli_ret:
        err_msg = get_common_msg(lang, "ssh.remote.failure")
        return False, all_cli_ret, err_msg

    return True, all_cli_ret, err_msg


def get_system_version_without_ret(cli_ret):
    hot_patch_begin_index = cli_ret.find("HotPatch Version")
    software_begin_index = cli_ret.find("Software Version")
    if software_begin_index == -1 or hot_patch_begin_index == -1:
        return False, "", ""
    software_version = get_horizontal_cli_ret(cli_ret[software_begin_index:hot_patch_begin_index])[0].\
        get("Current Version")
    hot_patch_version = get_horizontal_cli_ret(cli_ret[hot_patch_begin_index:])[0].get("Current Version")
    if hot_patch_version == "--":
        hot_patch_version = "0"
    return True, software_version, hot_patch_version
