# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

from java.lang import Exception as JException

from psdk.platform.protocol.cli_service import enter_target_mode, exit_to_developer_from_diagnose
from psdk.platform.entity.base_context import BaseContext
from psdk.platform.base.exception import SshConnectionException
from psdk.platform.util.base_util import ignore_all_exception
from psdk.platform.protocol.cli_factory import check_connect, create_cli_conn
from psdk.platform.protocol.cli_common import get_controller_ip_list

"""
屏蔽掉SVP内外差异，获取ctrl的连接
contextUtil.getCliCommon()-从上下文获取连接
contextUtil.getCli()--先从python的缓存中获取连接，没有就新建
contextUtil.get_new_cli()--先从python的缓存中获取连接，没有就新建
get_ctrl_cli() -针对高端设备， 返回控制器的连接。
针对需要多个连接的场景的使用方式：
遍历控制器使用连接的方式：
sftp_controller

"""
CLI_CONNECTION = "CLI_CONNECTION"
SVP_CTRL_CONNECTION = "SVP_CTRL_CONNECTION"


def get_ctrl_cli(context, cli=None, allow_not_admin=False):
    """
    屏蔽SVP设备的差异，获取到控制器的连接，支持以设备IP+sn作为缓存

    :param context: 上下文对象
    :param cli: 原始cli
    :param allow_not_admin: 是否允许获取到处于非admin状态的连接
    :return:
    """
    logger = context.get_logger()
    if context.get_dev_node().has_svp_module:
        logger.info("get_ctrl_cli has svp.")
        return get_conn_for_svp(context, cli, allow_not_admin)
    else:
        logger.info("get_ctrl_cli no svp")
        return get_available_conn(context, cli, allow_not_admin)


def get_available_conn(context, cli=None, allow_not_admin=False):
    """
    有连接，检查连接是否可用，不可用则重连
    没有连接，新建连接
    :param context:
    :param cli:
    :param allow_not_admin:
    :return:
    """
    dev_obj = context.get_dev_node()
    logger = context.get_logger()
    if cli:
        return check_connect(cli, logger, allow_not_admin)
    ip = dev_obj.get_ip()
    sn = dev_obj.get_sn()
    conn_key = CLI_CONNECTION + ip + sn
    conn = context.get_item(conn_key)
    logger.info("conn_key={}".format(conn_key))
    if not conn:
        logger.info("conn is None")
        conn = create_cli_conn(context, ip)
        conn = check_connect_in_correct_mode(conn, logger, allow_not_admin)
        context.set_item(conn_key, conn)
    else:
        conn = check_connect(conn, logger, allow_not_admin)
    return conn


@ignore_all_exception
def re_conn_silence(conn):
    conn.reConnect()


def check_connect_in_correct_mode(conn, logger, allow_not_admin=False):
    """
    获取连接并检查有效性

    :param conn: 原连接
    :param logger: logger
    :param allow_not_admin: 是否允许获取到的连接是非admin模式的，如果不允许则非admin模式时需要重连接以回到admin模式
    :return:
    """
    if conn is None:
        raise SshConnectionException("con is none")

    if not conn.isConnected():
        if logger:
            logger.debug("Ssh connection not connected, "
                         "reconnecting...")
        re_conn_silence(conn)
    if allow_not_admin:
        return conn
    try:
        check_conn_normal(conn)
    except (Exception, JException) as ex:
        logger.error("check_conn_error:{}".format(str(ex)))
        conn.reConnect()
    return conn


def check_conn_normal(conn):
    """
    当获取到的cli是在Storage模式时，进行重连
    :param conn: 连接
    :return: 如果在Storage模式raise异常。
    """
    cli_ret = conn.execCmd("show cli configuration")
    if "Storage:~" in cli_ret:
        raise SshConnectionException("in storage mode")


def get_conn_for_svp(context, cli, allow_not_admin=False):
    """
    针对SVP环境获取控制器连接并进行缓存

    :param context:
    :param cli:
    :param allow_not_admin:
    :return:
    """
    dev_obj = context.get_dev_node()
    logger = context.get_logger()
    conn = _get_conn_by_svp(context, logger, cli)
    logger.info("_get_conn_by_svp={}, cli is conn:{}".format(cli, bool(cli is conn)))
    if cli and cli is conn:
        logger.info("_get_conn_by_svp use cached cli.")
        return cli

    ip = dev_obj.get_ip()
    sn = dev_obj.get_sn()
    conn_key = CLI_CONNECTION + ip + sn + "_ctrl"
    conn = context.get_item(conn_key)
    logger.info("conn_key={}".format(conn_key))
    if not conn:
        logger.info("conn is None")
        conn = _get_conn_by_svp(context, logger, cli)
        conn = check_connect(conn, logger, allow_not_admin)
        context.set_item(conn_key, conn)
    else:
        conn = check_connect(conn, logger, allow_not_admin)
    return conn


def get_available_conn_svp(context, cli=None, allow_not_admin=False):
    """
    SVP 有连接，检查连接是否可用，不可用则重连
    没有连接，新建连接
    :param context:
    :param cli:
    :param allow_not_admin:
    :return:
    """
    dev_obj = context.get_dev_node()
    logger = context.get_logger()
    if cli:
        return check_connect(cli, logger, allow_not_admin)
    ip = dev_obj.get_ip()
    sn = dev_obj.get_sn()
    conn_key = SVP_CTRL_CONNECTION + ip + sn
    conn = context.get_item(conn_key)
    if conn:
        logger.info("get_available_conn_svp conn_key={}, conn:{}, is connect:{}".format(conn_key, conn, conn.isConnected()))
        return check_connect(conn, logger, allow_not_admin)

    cli = get_available_conn(context)
    lang = context.get_lang()
    flag, controller_ip_list = get_controller_ip_list(cli, lang)
    if flag:
        # 返回第一个成功连接的控制器
        for controller_ip in controller_ip_list:
            cli = create_cli_conn(context, controller_ip)
            if cli:
                logger.debug("init cli connection successfully")
                break
    # 替换连接key为控制器直连连接
    logger.info("new available conn svp conn_key={}".format(conn_key))
    context.set_item(conn_key, cli)
    return cli


def _get_conn_by_svp(context, logger, cli=None):
    """
    高端SVP环境获取控制器的cli连接
    有连接，检查连接是否正常以及能否进入debug模式，如果能就直接返回
    当前没有连接，则重新建立连接，如果连接无法进入debug模式，还需要建立到控制器的连接
    """
    # 有svp 先获取控制器的ip 然后创建连接到该ip的连接
    if not cli:
        logger.debug("no cli, get conn by context cli:{}".format(cli))
        cli = get_available_conn_svp(context, cli)
    if not cli.isConnected():
        cli = check_connect(cli, logger, False)
    logger.debug("_get_conn_by_svp:{}, cli:{}".format(cli.isConnected(), cli))
    # 检查cli是否可以进入debug
    if enter_target_mode(cli, "diagnose", "en")[0]:
        logger.debug("current cli can enter debug mode.")
        exit_to_developer_from_diagnose(cli, "en")
        return cli
    lang = context.get_lang()
    flag, controller_ip_list = get_controller_ip_list(cli, lang)
    if flag:
        # 返回第一个成功连接的控制器
        for controller_ip in controller_ip_list:
            cli = create_cli_conn(context, controller_ip)
            if cli:
                logger.debug("init cli connection successfully")
                break
    # 替换连接key为控制器直连连接
    dev_obj = context.get_dev_node()
    ip = dev_obj.get_ip()
    sn = dev_obj.get_sn()
    conn_key = SVP_CTRL_CONNECTION + ip + sn
    logger.info("_get_conn_by_svp conn_key={}".format(conn_key))
    context.set_item(conn_key, cli)
    return cli


def create_cli_connection(context, ip):
    """根据ip创建cli连接，支持跳转

    :param context: 上下文对象
    :param ip: 创建cli连接的ip
    :param set_item 是否将连接设置到context中保存
    :return: 创建成功返回cli连接，否则为None
    """
    return create_cli_conn(context, ip)


def close_cli_connection(cli):
    """
    @summary: 关闭cli连接
    @param cli:cli连接
    """
    cli.close()



