# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.


from psdk.platform.protocol import cli_util


class CliMode:
    CLI = 'cli'
    DEVELOPER = 'developer'
    DIAGNOSE = 'diagnose'
    MINI_SYSTEM = 'mini_system'


def exec_cli(cli, cmd, lang, **kwargs):
    return exec_command(cli, cmd, CliMode.CLI, lang, **kwargs)


def exec_developer(cli, cmd, lang, debug_pass=None, **kwargs):
    return exec_command(cli, cmd, CliMode.DEVELOPER, lang, debug_pass, **kwargs)


def exec_diagnose(cli, cmd, lang, debug_pass=None, **kwargs):
    return exec_command(cli, cmd, CliMode.DIAGNOSE, lang, debug_pass, **kwargs)


def exec_mini_system(cli, cmd, lang, debug_pass=None, **kwargs):
    return exec_command(cli, cmd, CliMode.MINI_SYSTEM, lang, debug_pass, **kwargs)


def enter_developer_from_cli(cli, lang, **k_args):
    """
    Entering the developer mode from the admin mode
    """
    is_open, execute_ret = cli_util.needOpenDeveloperSwitch(cli, lang)
    if is_open is True:
        execute_ret = cli_util.openDeveloperSwitch(cli, lang)
        if not execute_ret[0]:
            return execute_ret
    return cli_util.enter_developer_from_cli(cli, lang, k_args.get('debug_pass'))


def enter_diagnose_from_developer(cli, lang):
    execute_ret = cli_util.execute_cmd(cli, 'debug', True)
    if not execute_ret[0]:
        return execute_ret
    if cli_util.isInDebugMode(execute_ret[1]):
        return True, execute_ret[1]
    return False, execute_ret[1]


def enter_mini_sys_from_developer(cli, lang):
    cli_ret = cli.execCmd("minisystem")
    if cli_util.isInMinisystemMode(cli_ret):
        return True, cli_ret
    if not cli_util.hasCliExecPrivilege(cli_ret):
        return False, cli_ret
    return False, cli_ret


def exit_to_developer_from_diagnose(cli, lang):
    execute_ret = cli_util.execute_cmd(cli, "exit")
    if "(y/n)" in execute_ret[1]:
        execute_ret = cli_util.execute_cmd(cli, "y")
    return cli_util.isInDeveloperMode(execute_ret[1]), execute_ret[1]


def exit_to_developer_from_mini_sys(cli, lang):
    execute_ret = cli_util.execute_cmd(cli, "exit")
    if "(y/n)" in execute_ret[1]:
        execute_ret = cli_util.execute_cmd(cli, "y")
    return cli_util.isInDeveloperMode(execute_ret[1]), execute_ret[1]


def exec_command(cli, cmd, target_mode, lang, debug_pass=None, **kwargs):
    flag, cli_res = enter_target_mode(cli, target_mode, lang, debug_pass)
    if flag is not True:
        return flag, cli_res
    return cli_util.execute_cmd(cli, cmd, **kwargs)


def enter_target_mode(cli, target_mode, lang, debug_pass=None):
    current_mode, cli_res, error = get_current_mode(cli, lang)
    if not current_mode:
        return False, cli_res
    if target_mode != current_mode:
        enter_mode_func_list = MODE_FUNC_DICT.get(current_mode + target_mode)
        for func in enter_mode_func_list:
            if func == enter_developer_from_cli:
                flag, cli_res = func(cli, lang, debug_pass=debug_pass)
            else:
                flag, cli_res = func(cli, lang)
            if flag is not True:
                return flag, cli_res
    return True, ''


MODE_DICT = [
    {'mode': CliMode.DEVELOPER, 'end_str': ['developer:/>']},
    {'mode': CliMode.DIAGNOSE, 'end_str': ['/diagnose>']},
    {'mode': CliMode.MINI_SYSTEM, 'end_str': ['minisystem>']},
]


def get_current_mode(cli, lang):
    flag, res = cli_util.execute_cmd(cli, 'show cli configuration')
    if flag is not True:
        return None, res
    res = res.strip()
    for one_mode in MODE_DICT:
        for end_str in one_mode['end_str']:
            if res.endswith(end_str):
                return one_mode['mode'], res, ''
    return CliMode.CLI, res, ''


MODE_FUNC_DICT = {
    CliMode.CLI + CliMode.DEVELOPER: [
        enter_developer_from_cli
    ],
    CliMode.CLI + CliMode.DIAGNOSE: [
        enter_developer_from_cli,
        enter_diagnose_from_developer
    ],
    CliMode.CLI + CliMode.MINI_SYSTEM: [
        enter_developer_from_cli,
        enter_mini_sys_from_developer
    ],
    CliMode.DEVELOPER + CliMode.CLI: [
    ],
    CliMode.DEVELOPER + CliMode.DIAGNOSE: [
        enter_diagnose_from_developer
    ],
    CliMode.DEVELOPER + CliMode.MINI_SYSTEM: [
        enter_mini_sys_from_developer
    ],
    CliMode.DIAGNOSE + CliMode.CLI: [
        exit_to_developer_from_diagnose
    ],
    CliMode.DIAGNOSE + CliMode.DEVELOPER: [
        exit_to_developer_from_diagnose
    ],
    CliMode.DIAGNOSE + CliMode.MINI_SYSTEM: [
        exit_to_developer_from_diagnose,
        enter_mini_sys_from_developer
    ],
    CliMode.MINI_SYSTEM + CliMode.CLI: [
        exit_to_developer_from_mini_sys
    ],
    CliMode.MINI_SYSTEM + CliMode.DEVELOPER: [
        exit_to_developer_from_mini_sys
    ],
    CliMode.MINI_SYSTEM + CliMode.DIAGNOSE: [
        exit_to_developer_from_mini_sys,
        enter_diagnose_from_developer
    ]
}
