# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import re

CLI_RET_END_FLAG = ":/>"
DIAGNOSE_END_FLAG = "diagnose>"
MINISYSTEM_END_FLAG = "minisystem>"


def get_vertical_cli_ret(cli_ret):
    """
    按逐行字典的方式获取垂直表格形式的cli回显集合

    :param cli_ret:
    :return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    cli_ret_list = cli_ret.splitlines()
    dict_list = []
    line_dict = {}

    for line in cli_ret_list:
        line = str(line)
        if CLI_RET_END_FLAG in line:
            break

        if re.search("^-+\r*\n*$", line):
            dict_list.append(line_dict.copy())
            line_dict.clear()

        fields = line.split(":")
        if len(fields) < 2:
            continue

        key = fields[0].strip()
        value = ":".join(fields[1:len(fields)]).strip()

        if key in line_dict:
            key += "_" + str(str(line_dict.keys()).count(key + "_") + 1)
        line_dict.setdefault(key, value)

    if len(line_dict) > 0:
        dict_list.append(line_dict.copy())

    return dict_list


def get_horizontal_cli_ret(cli_ret):
    """
    按逐行字典的方式获取水平表格形式的cli回显集合

    :param cli_ret:
    :return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    try:
        headline = ""
        i = 0
        cli_ret_list = cli_ret.splitlines()
        for line in cli_ret_list:
            line = str(line)
            match = re.search(r"^\s*-+(\s+-+)*\s*$", line)
            if match:
                headline = match.group(0)
                break
            i += 1

        if headline == "" or i == 0 or i >= len(cli_ret_list) - 1:
            return []

        title = cli_ret_list[i - 1]
        field_words = cli_ret_list[(i + 1):]
        tuple_index = get_start_end_list(r"\s*-+\s*", headline)

        keys = []
        for item in tuple_index:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key)

        required_line_len = tuple_index[-1][0]
        dict_list = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break

            if len(line.strip()) == 0:
                continue

            if len(line) <= required_line_len:
                continue

            values = _parse_by_double_space(line)
            if len(values) == len(keys):
                dict_list.append(dict(zip(keys, values)))
                continue

            values = _parse_by_sector_index(line, tuple_index)
            dict_list.append(dict(zip(keys, values)))

        return dict_list
    except Exception:
        return []


def get_horizontal_cli_ret_compatible_chinese(cli_ret):
    """
    按逐行字典的方式获取水平表格形式的cli回显集合
    可以适用于中文场景
    :param cli_ret: 回文
    :return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    headline = ""
    i = 0
    cli_ret_list = cli_ret.splitlines()
    for line in cli_ret_list:
        line = str(line)
        match = re.search(r"^\s*-+(\s+-+)*\s*$", line)
        if match:
            headline = match.group(0)
            break
        i += 1

    if headline == "" or i == 0 or i >= len(cli_ret_list) - 1:
        return []

    title = cli_ret_list[i - 1]
    field_words = cli_ret_list[(i + 1):]
    tuple_index = get_start_end_list(r"\s*-+\s*", headline)

    keys = []
    for item in tuple_index:
        key = title[item[0]:item[1]].strip()
        if keys.count(key):
            key += "_" + str(str(keys).count(key + "_") + 1)
        keys.append(key)

    required_line_len = tuple_index[-1][0]
    dict_list = []
    for line in field_words:
        if CLI_RET_END_FLAG in line or line.strip().endswith(DIAGNOSE_END_FLAG) or line.strip().endswith(
                MINISYSTEM_END_FLAG):
            break

        if len(line.strip()) == 0:
            continue

        values = _parse_by_double_space(line)
        if len(values) < len(keys):
            continue

        if len(values) == len(keys):
            dict_list.append(dict(zip(keys, values)))
            continue

        values = _parse_by_sector_index(line, tuple_index)
        dict_list.append(dict(zip(keys, values)))

    return dict_list


def get_start_end_list(regex, s, flags=0):
    """
    使用正则表达式匹配，返回所有匹配结果的开始结尾字典
    :param regex: 正则表达式
    :param s: 要匹配的字符串
    :return: 所有匹配字符串的开头结尾list
    """

    reg = re.compile(regex, flags)
    tuple_idxs = []
    items = reg.finditer(s)
    for item in items:
        tuple_idxs.append(item.regs[0])

    return tuple_idxs


def horizontal_parser_for_diagnose(
        ret, key_list, end_flag=DIAGNOSE_END_FLAG, split_key=None):
    """
        专门处理表头不是用空格区分的，而是制表符区分的场景，
        正常按位无法解析的场景。使用python自带的split划分表头和值,
        注意：不能用于处理值中有空值的场景。
        :param ret: 回文
        :param key_list: 需要解析的字段
        :param end_flag: 结束符
        :param split_key: title分隔符
        :return: 字典列表
        """

    try:
        res_dict_list = __horizontal_parser_for_diagnose_impl(
            end_flag, key_list, ret, split_key)
    except Exception:
        return res_dict_list
    return res_dict_list


def __horizontal_parser_for_diagnose_impl(end_flag, key_list, ret, split_key):
    hor_header_dict = {}
    res_dict_list = []
    ret_list = ret.splitlines()
    title_find_flag = False
    for ret in ret_list:
        if not title_find_flag and __check_all_key_in_line(ret, key_list):
            if not split_key:
                head_keys = ret.split()
            else:
                head_keys = list(
                    map(lambda t: t.strip(), ret.split(split_key))
                )
            for key in key_list:
                hor_header_dict[key] = head_keys.index(key)
            title_find_flag = True
            continue

        # 排除 title下面为-- 的空行
        if not ret.replace("-", "").strip():
            continue

        # 解析数据行
        if title_find_flag and not ret.endswith(end_flag):
            res_dict = {}
            values = ret.split()
            if len(values) < len(hor_header_dict):
                continue
            for key in hor_header_dict:
                value_index = hor_header_dict.get(key)
                res_dict[key] = values[value_index]
            res_dict_list.append(res_dict)
    return res_dict_list


def __check_all_key_in_line(line, keys):
    """
    检查是否所有的key都在此行中，如果都在话判定为True
    :param line: 待确认行
    :param keys: 字段
    :return: True 是， False 不是
    """
    for key in keys:
        if key not in line:
            return False
    return True


def _parse_by_double_space(line):
    values = []
    value_list = line.strip().split("  ")
    for value in value_list:
        if value != "":
            values.append(value.strip())
    return values


def _parse_by_sector_index(line, tuple_index):
    values = []
    for item in tuple_index:
        values.append(line[item[0]:item[1]].strip())
    return values
