# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.
import datetime

import cliUtil
import common
import dpa_common

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)
# 接口返回列表最多个数
MAX_CONTENT_COUNT = 200
HOUR = 60
MINUTE = 60
SECOND = 1000
NODE_QUERY_URI = "/backups/backup_job/instances?backupType=&count={}&index={}&is_all=1&status=1,4,64,128,256,512"


def execute(dpa_rest):
    time_stamp, record = dpa_common.get_time_stamp(dpa_rest, PY_JAVA_ENV, LOGGER)
    if not time_stamp:
        # 未添加DPA对应主机或回显不符合预期
        return cliUtil.RESULT_NOCHECK, record, common.getMsg(LANG, "dpa.node.select.abnormal")
    flag, status_record, err_msg = check_historical_tasks_status(dpa_rest, time_stamp)
    record = "{}\n{}".format(record, status_record)
    return flag, record, err_msg


def check_historical_tasks_status(dpa_rest, time_stamp):
    """
    循环查询
    可能有10000+记录
    """
    err_msg = ""
    index = 0
    record = ""
    try:
        record = dpa_common.dpa_rest_pagination_query(dpa_rest, NODE_QUERY_URI, index, MAX_CONTENT_COUNT)
        dev_historical_tasks = record.get("responseData", {}).get("data", [{}])
        total_num = record.get("responseData", {}).get("totalNum")
        while total_num > index + MAX_CONTENT_COUNT:
            index += MAX_CONTENT_COUNT
            tem_record = dpa_common.dpa_rest_pagination_query(dpa_rest, NODE_QUERY_URI, index, MAX_CONTENT_COUNT)
            record = "{}\n{}".format(record, tem_record)
            dev_historical_tasks.extend(tem_record.get("responseData", {}).get("data", [{}]))

        abnormal_backup_task_dict = filter_abnormal_task(dev_historical_tasks, time_stamp)
        if not abnormal_backup_task_dict:
            return True, record, err_msg

        abnormal_backup_task_list = sort_abnormal_task(abnormal_backup_task_dict)
        err_msg += dpa_common.get_dpa_msg(LANG, "dpa.historical.tasks.status.abnormal")
        for task in abnormal_backup_task_list:
            err_msg += dpa_common.get_dpa_msg(LANG, "dpa.historical.tasks.status.abnormal.record",
                                              (task[0], str(task[1]), datetime.datetime.fromtimestamp(task[2] // 1000)))
        return False, record, err_msg

    except Exception as e:
        LOGGER.logError(str(e))
        return cliUtil.RESULT_NOCHECK, record, common.getMsg(LANG, "query.result.abnormal")


def filter_abnormal_task(dev_historical_tasks, time_stamp):
    """
    筛选出状态异常的备份任务，同一名称的备份任务，每一种异常状态仅保存最近一次的执行记录
    :param dev_historical_tasks: 所有备份任务
    :param time_stamp: 设备当前时间
    :return: 状态异常的备份任务
            {
                "task_name_1": {
                    "task_status_1": "task_start_time",
                    "task_status_2": "task_start_time",
                },
                "task_name_2": {
                    "task_status_1": "task_start_time",
                    "task_status_2": "task_start_time",
                },
                ...
            }
    """
    abnormal_backup_task_dict = {}
    for task in dev_historical_tasks:
        task_name = task.get("instJobName").decode("unicode_escape")
        task_status = task.get("instStatus")
        task_start_time = task.get("instStartTime")
        # 值为256（异常）或64（失败），则检查不通过
        abnormal_standard_1 = task_status in (256, 64)
        # 值为1（准备中）或4（执行中）或512（停止中），且其instUpdateTime字段值为超过一天，即小于等于当前设备时间减去24小时，则检查不通过
        abnormal_standard_2 = task_status in (1, 4, 512) and int(time_stamp) - task.get(
            "instUpdateTime") > 24 * HOUR * MINUTE * SECOND
        # 值为128（部分成功），且其instJobType不为“eso_backupengine_fileengine”，则检查不通过
        abnormal_standard_3 = task_status == 128 and task.get("instJobType") != "eso_backupengine_fileengine"
        if abnormal_standard_1 or abnormal_standard_2 or abnormal_standard_3:
            # 该名称备份任务最近一次出现相同状态异常的开始时间
            latest_task_start_time = abnormal_backup_task_dict.get(task_name, {}).get(task_status, 0)
            abnormal_backup_task_dict.setdefault(task_name, {})[task_status] = \
                task_start_time if task_start_time > latest_task_start_time else latest_task_start_time
    return abnormal_backup_task_dict


def sort_abnormal_task(abnormal_backup_task_dict):
    """
    将状态异常的备份任务按照任务开始时间从高到低排序
    :param abnormal_backup_task_dict: 状态异常的备份任务
    :return: 将状态异常的备份任务按照任务开始时间从高到低排序后的列表
            [
                (task_name, task_status, task_start_time),
                (task_name, task_status, task_start_time),
                ...
            ]
    """
    abnormal_backup_task_list = []
    for task_name, task_detail in abnormal_backup_task_dict.items():
        for task_status, task_latest_start_time in task_detail.items():
            abnormal_backup_task_list.append((task_name, task_status, task_latest_start_time))
    abnormal_backup_task_list.sort(key=lambda x: x[2], reverse=True)
    return abnormal_backup_task_list
