# -*- coding: UTF-8 -*-
import common
import common_cache
import cliUtil
from common_utils import get_err_msg
from common import UnCheckException

RISK_VERSION = "V300R006C00SPC100"
DEVICE_MODEL = (
    "5300 V3",
    "5500 V3",
    "5600 V3",
    "5800 V3",
    "6800 V3",
    "6900 V3",
    "2600 V3",
    "2600 V3 for Video",
    "2800 V3",
)


def execute(cli):
    """
    高密硬盘框类型检查
    检查方法(中文):
    步骤1 以admin用户登录设备。
    步骤2 执行命令show system general获取设备型号。
    步骤3 执行命令show upgrade package获取设备软件版本。
    检查标准(中文):
    1. 若新扩容硬盘框包含12GB高密硬盘框，则继续检查，否则检查结果为通过；
    2. 新扩12G高密硬盘框，设备类型为5300 V3/5500 V3/5600 V3/5800 V3/6800 V3/6900
     V3/2600 V3/2600 V3 for Video/2800 V3且版本低于V300R006C00SPC100，则检查不通
     过，否则检查通过；
    错误提示：
    当前软件版本为XXX,不支持扩容高密硬盘框。
    修复建议：
    请升级到V300R006C00SPC100及以上版本后，再进行扩容硬盘框。
    """

    logger = common.getLogger(PY_LOGGER, __file__)
    lang = common.getLang(py_java_env)
    logger.logInfo("check_high_density_disk_enclosure_type---")
    is_12high = py_java_env.get("include12HighEnclosure")
    enclosure_check = HighDensityDiskEnclosureTypeCheck(
        cli, lang, logger, is_12high
    )
    flag, err_msg = enclosure_check.execute_check()

    return flag, "\n".join(enclosure_check.all_cli_ret), err_msg


class HighDensityDiskEnclosureTypeCheck(object):
    def __init__(self, cli, lang, logger, is_12high):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.is_12high = is_12high
        self.all_cli_ret = list()

    def execute_check(self):

        try:
            # 新扩硬盘框是否包含12G高密硬盘框
            if self.is_12high == "no":
                return True, ""
            # 检查版本
            return self.check_version()

        except UnCheckException as uncheck:
            return cliUtil.RESULT_NOCHECK, uncheck.errorMsg
        except Exception:
            err_msg = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(self.lang, err_msg)

    def check_version(self):
        """
        检查软件版本
        :return:True: 检查通过
                False: 检查不通过
        """

        (
            flag,
            cli_ret,
            err_msg,
            dev_model,
        ) = common_cache.get_product_model_with_cliret_cache(
            py_java_env, self.cli, self.logger
        )
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg)

        (
            flag,
            dev_version,
            p_patch,
            cli_ret,
            err_msg,
        ) = common_cache.get_version_and_patch_cache(
            py_java_env, self.cli, self.logger
        )
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg)

        self.logger.logInfo(
            "dev_version is: {}; dev_model is: {}".format(
                dev_version, dev_model
            )
        )
        if dev_model in DEVICE_MODEL and dev_version < RISK_VERSION:
            err_msg = "expand.enclosure.version.check.risk"
            return False, get_err_msg(self.lang, err_msg, (dev_version,))
        return True, ""
