# -*- coding: UTF-8 -*-
import traceback

import cliUtil
import common
import common_cache
from storage_obj_constant import InterfaceModule as InterfaceAttr

from common_utils import get_err_msg
from cli_util_cache import get_horizontal_cache

from cbb.frame.base.config import (
    DORADO_DEVS_V6_HIGH,
    DORADO_DEVS_V6_MID,
    DORADO_DEVS_V6_LOW,
)

DEV_LOWER = ["IOM1"]
DEV_MID = ["IOM2"]
DEV_HIGH = ["H3", "L3", "H10", "L10"]
BE_TYPE_KEY = " BE "
RDMA_TYPE = "RDMA"

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    ScaleOut槽位卡类型检查
    :param cli:
    :return:
    """
    scaleout_slot_card_check = ScaleoutSlotCardCheck(
        cli, LANG, PY_JAVA_ENV, LOGGER
    )
    flag, msg = scaleout_slot_card_check.execute_check()
    return flag, "\n".join(scaleout_slot_card_check.all_ret_list), msg


class ScaleoutSlotCardCheck:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []
        self.err_msg = ""
        self.p_version = ""
        self.interface_info_list = []
        self.interface_slot_key = ""
        self.product_mode = str(env.get("devInfo").getDeviceType())

    def execute_check(self):
        try:
            if self.is_two_to_four_ctrl():
                return True, ""

            flag, msg = self.get_all_information()
            if flag is not True:
                return flag, msg

            risk_msg_list = self.get_risk_interface_module()
            if risk_msg_list:
                return False, "".join(risk_msg_list)

            return True, ""
        except common.UnCheckException as e:
            LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            LOGGER.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def is_two_to_four_ctrl(self):
        """
        引擎间，高端2扩4，检查通过。
        :return:
        """
        dev_ctrl_num = int(self.env.get("expInfo").getDevCtrlsNum())
        exp_ctrl_num = int(self.env.get("expInfo").getExpCtrlsNum())
        return all(
            (
                self.product_mode in DORADO_DEVS_V6_HIGH,
                dev_ctrl_num == 2,
                exp_ctrl_num == 4,
            )
        )

    def get_interface_slot_key(self):
        """
        获取各层级检查的槽位
        :return:槽位
        """
        # 高端：3号槽位和10号槽位（H3,L3,H10,L10）
        if self.product_mode in DORADO_DEVS_V6_HIGH:
            return DEV_HIGH
        # 中端：2号槽位（IOM2）
        if self.product_mode in DORADO_DEVS_V6_MID:
            return DEV_MID
        # 低端：1号槽位（IOM1）
        if self.product_mode in DORADO_DEVS_V6_LOW:
            return DEV_LOWER

    def get_risk_interface_module(self):
        """
        任一Scaleout槽位卡类型为BE类型且扩容场景为引擎间扩控，
        则检查结果为不通过
        :return: 风险接口卡
        """
        risk_interface_msg = []
        for record in self.interface_info_list:
            interface_id = record.get(InterfaceAttr.ID, "")
            interface_model = record.get(InterfaceAttr.MODEL, "")
            if not (
                BE_TYPE_KEY in interface_model
                and RDMA_TYPE in interface_model
            ):
                continue

            for interface_key in self.interface_slot_key:
                if interface_id.endswith(interface_key):
                    msg_key = "exp.scalout.slot.card.type.not.pass"
                    risk_interface_msg.append(
                        get_err_msg(
                            self.lang,
                            msg_key,
                            (self.p_version, interface_id, interface_model),
                        )
                    )
        return risk_interface_msg

    def get_all_information(self):
        """
        获取设备当前系统软件版本, 获取Scaleout槽位卡类型
        :return:
        """
        (
            flag,
            self.p_version,
            p_patch,
            ret,
            msg,
        ) = common_cache.get_version_and_patch_cache(
            self.env, self.cli, self.logger
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            return flag, msg

        cmd = "show interface_module"
        flag, ret, self.interface_info_list, msg = get_horizontal_cache(
            self.cli, self.env, self.logger, cmd
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            return flag, msg

        self.interface_slot_key = self.get_interface_slot_key()

        return True, ""
