# -*- coding: UTF-8 -*-

from cbb.frame.checkitem.context_adapter import InspectContext
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.base import baseUtil
from cbb.frame.base import product
from cbb.frame.cli import cliUtil
from cbb.frame.base.exception import UnCheckException
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN
from utils import Products

import common
import xlrd
import os

PY_JAVA_ENV = py_java_env
JAVA_LOGGER = PY_LOGGER
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(JAVA_LOGGER, __file__)
ALARM_THRESHOLD_FILE_NAME = "optical_module_power_alarm_threshold.xls"
PORT_ID = "PortID"
TX_PW_KEY = ["txpowerreal(uw)", "txpower(uw)", "txpower"]
RX_PW_KEY = ["rxpowerreal(uw)", "rxpower(uw)", "rxpower"]
# 进度刷新间隔
INTERVAL = 2
# 进度条刷新
LIMIT_TIME = 120

def execute(cli):
    all_ret = ""

    try:
        LOGGER.logInfo("check_item_hardware_optical_module_power start.")
        # 进度条刷新
        common.threadUpProcess(PY_JAVA_ENV, LIMIT_TIME, INTERVAL, LOGGER)
        # 进度开始
        common.inProcess(PY_JAVA_ENV)
        context = InspectContext(PY_JAVA_ENV, cli)
        inspect_context = context.get_context()
        inspect_context["logger"] = JAVA_LOGGER
        inspect_context["lang"] = LANG
        check_obj = CheckItem(inspect_context, cli)
        status, err_msg = check_obj.check()
        all_ret = check_obj.get_origin_info()
        LOGGER.logInfo("status={},err_msg={}".format(status, err_msg))
        return context.get_result(status, err_msg, all_ret)
    finally:
        common.finishProcess(PY_JAVA_ENV)
        LOGGER.logInfo("finish process!")


class CheckItem(BaseCheckItem):
    def __init__(self, context, cli):
        super(CheckItem, self).__init__(context)
        self.dev_node = context.get("devInfo")
        self.context[NEED_NEW_CTRL_CONN] = False
        self.dev_version = ""
        self.patch_version = ""
        self.cli = cli
        self.env = PY_JAVA_ENV
        self.dev_type = str(self.dev_node.getDeviceType())

    def execute(self):
        LOGGER.logInfo("Optical_module_power_inspection start.")
        self.init_version_and_patch()
        LOGGER.logInfo(
            "dev_version:{} patch_version:{}.".format(self.dev_version,
                                                      self.patch_version))
        # 1、若步骤2中版本满足以下条件则继续检查，否则检查结果为不涉及；
        if not self.is_risk_version():
            return CheckStatus.NO_SUPPORT, ""
        # 2、若步骤3中不存在运行状态为Link Up的光模块则检查结果为通过，否则继续检查
        flag, port_id_list = self.get_link_up_fibre_port_id()
        LOGGER.logInfo("port_id_list:{}".format(port_id_list))
        if not flag:
            return CheckStatus.PASS, ""
        # 3、若步骤4中Model值都不在告警门限列表中则检查结果为通过，否则继续检查；
        # 4、若步骤4中RxPowerReal(uW)、TxPowerReal(uW)或者Rx power(uW)、Tx power(uW)
        # 任一值低于告警门限值则检查结果为建议优化，否则检查结果为通过。

        result_list = self.check_fibre_model_power(port_id_list)
        if not result_list:
            return CheckStatus.PASS, ""

        err_msg = baseUtil.getPyResource(self.lang,
                                         "check.Optical.module.power.warming",
                                         "\n".join(result_list))
        return CheckStatus.WARNING, err_msg

    def init_version_and_patch(self):
        """
        初始化版本、补丁信息
        """
        (
            flag,
            self.dev_version,
            self.patch_version,
            ret,
        ) = replace_adapter.get_system_version_with_ret(self.context)
        self.add_origin_info(ret)
        if flag is not True:
            raise UnCheckException("Failed to get system version.")

    def is_risk_version(self):
        """
        1、若步骤2中版本满足以下条件则继续检查，否则检查结果为不涉及；
         1）若系统版本为V3R6C00SPC100或V500R007C00或之后版本；
         2）若系统版本为6.1.0或之后版本；
         3）若系统版本为V500R007C60SPC200 Kunpeng或之后版本；
        :return: 是否为风险版本。
        """
        if "V3" in self.dev_version and self.dev_version >= "V300R006C00SPC100":
            return True
        # 判断kunpeng
        product_version = self.dev_node.getProductVersion()
        LOGGER.logInfo("is_risk_version product_version:{}".format(product_version))
        is_kunpeng = product.isKunpeng(str(product_version))
        if "V5" in self.dev_version:
            if not is_kunpeng and self.dev_version >= "V500R007C00":
                return True
            if is_kunpeng and self.dev_version >= "V500R007C60SPC200":
                return True
        is_dorado_v6 = baseUtil.isDoradoV6Dev(self.dev_type)
        if is_dorado_v6 and Products.compareVersion(self.dev_version, "6.1.0") >= 0:
            return True
        return False

    def get_link_up_fibre_port_id(self):
        """
        步骤3 执行命令：show port fibre_module|filterRow column=Running\sStatus predict=equal_to value=Link\sUp，
        获取运行状态为Link Up的光模块PortID；
        :return: True 结果, port_id_list返回PortID列表
        """
        port_id_list = []
        cmd = "show port fibre_module|filterRow column=Running\sStatus predict=equal_to value=Link\sUp"

        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.add_origin_info(ret)
        if flag is not True:
            raise UnCheckException(
                "Failed to get port id of Link Up fibre model . ")

        fibre_datas = cliUtil.getHorizontalCliRet(ret)
        if not fibre_datas:
            return False, port_id_list
        for fibre_data in fibre_datas:
            port_id = fibre_data.get(PORT_ID, "")
            port_id_list.append(port_id)
        return True, port_id_list

    def get_fibre_model_data(self, port_id):
        """
            步骤4 执行命令：show port fibre_module port_id=PortID，
            获取光模块功率RxPowerReal(uW)、TxPowerReal(uW)或者Rx power(uW)、Tx power(uW)和Model信息；
        :param port_id: 宽口
        :return: fibre_info[0]光模块信息。
        """

        cmd = "show port fibre_module port_id={}".format(port_id)

        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.add_origin_info(ret)
        if flag is not True:
            raise UnCheckException(
                "Failed to get data of Link Up fibre model by port id. ")
        fibre_info = cliUtil.getVerticalCliRet(ret)
        if not fibre_info:
            raise UnCheckException(
                "Failed to get data of Link Up fibre model by port id.")
        return fibre_info[0]

    def check_fibre_model_power(self, port_id_list):
        """
        3、若步骤4中Model值都不在告警门限列表中则检查结果为通过，否则继续检查；
        错误提示为：
        当前系统存在光模块功率偏低；
        光模块（PortID：XX）：RxPowerReal(uW)：XX（RX功率告警门限）、TxPowerReal(uW)：XX（TX功率告警门限）
        :param port_id_list: 光模块端口
        :return: 检查结果，需要优化光模块列表。
        """
        result_list = []
        # 获取光模块功率告警门限列表
        op_model_power_alarm_dict = Excel().get_excel_to_list()
        for port_id in port_id_list:
            fibre_data = self.get_fibre_model_data(port_id)
            model = fibre_data.get("Model").strip()
            if model not in op_model_power_alarm_dict:
                continue
            alarm_op_model = self.check_tx_rx_power(fibre_data, model,
                                                    op_model_power_alarm_dict)
            if alarm_op_model:
                err_msg = baseUtil.getPyResource(self.lang,
                                                 "check.Optical.module.port.id",
                                                 (port_id,
                                                  ",".join(alarm_op_model)))
                LOGGER.logInfo(
                    "check_fibre_model_power :{}.".format(err_msg))
                result_list.append(err_msg)
        return result_list

    def check_tx_rx_power(self, fibre_data, model, op_model_power_alarm_dict):
        new_fibre_info = self.trans_dict_key_to_lower(fibre_data)
        tx_pw = self.get_power_by_key(new_fibre_info, TX_PW_KEY)
        rx_pw = self.get_power_by_key(new_fibre_info, RX_PW_KEY)
        alarm_op_model = []
        tx_err = ""
        rx_err = ""
        flag, tx_err = self.check_multiple_tx_power(model, tx_pw,
                                                    op_model_power_alarm_dict)
        if not flag:
            alarm_op_model.append(tx_err)
        flag, rx_err = self.check_multiple_rx_power(model, rx_pw,
                                                    op_model_power_alarm_dict)
        if not flag:
            alarm_op_model.append(rx_err)

        return alarm_op_model

    def check_multiple_rx_power(self, model, multiple_rx_pw,
                                op_model_power_alarm_dict):
        rx_pw_arr = multiple_rx_pw.split(",")
        for rx_pw in rx_pw_arr:
            flag, err_msg = self.check_rx_power(model, rx_pw,
                                                op_model_power_alarm_dict)
            if not flag:
                return False, err_msg
        return True, ""

    def check_multiple_tx_power(self, model, multiple_tx_pw,
                                op_model_power_alarm_dict):
        tx_pw_arr = multiple_tx_pw.split(",")
        for tx_pw in tx_pw_arr:
            flag, err_msg = self.check_tx_power(model, tx_pw,
                                                op_model_power_alarm_dict)
            if not flag:
                return False, err_msg
        return True, ""

    def check_rx_power(self, model, rx_pw, op_model_power_alarm_dict):
        """
        功率的判断标准为10 =<power<门限值为建议优化，小于10、--，为通过。
        :param model: 型号
        :param rx_pw:接收实际功率
        :param op_model_power_alarm_dict: 光模块功率门限配置信息。
        :return:
        """
        if rx_pw == "--":
            return True, ""

        # 通过model获取对应的门限值
        power_alarm_threshold = op_model_power_alarm_dict.get(model)
        rx_pw_excel = power_alarm_threshold.get("RxPower")
        LOGGER.logInfo(
            "check_rx_power rx_pw_excel:{} rx_pw:{}.".format(rx_pw_excel,
                                                             rx_pw))
        rx_pw = float(rx_pw)
        if 10 <= rx_pw < float(rx_pw_excel):
            rx_err = baseUtil.getPyResource(self.lang,
                                            "check.Optical.module.power.rx.alarm",
                                            (rx_pw, rx_pw_excel))
            return False, rx_err
        return True, ""

    def check_tx_power(self, model, tx_pw, op_model_power_alarm_dict):
        """
        功率的判断标准为10 =<power<门限值为建议优化，小于10、--，为通过。
        :param model: 型号
        :param tx_pw:发送实际功率
        :param op_model_power_alarm_dict: 光模块功率门限配置信息。
        :return:
        """
        if tx_pw == "--":
            return True, ""

        # 通过model获取对应的门限值
        power_alarm_threshold = op_model_power_alarm_dict.get(model)
        tx_pw_excel = power_alarm_threshold.get("TxPower")
        LOGGER.logInfo(
            "check_tx_power tx_pw_excel:{} tx_pw:{}.".format(tx_pw_excel,
                                                             tx_pw))
        tx_pw = float(tx_pw)
        tx_pw_excel = float(tx_pw_excel)
        if 10 <= tx_pw < tx_pw_excel:
            tx_err = baseUtil.getPyResource(self.lang,
                                            "check.Optical.module.power.tx.alarm",
                                            (tx_pw, tx_pw_excel))
            return False, tx_err
        return True, ""

    def get_power_by_key(self, new_fibre_info, pw_key):
        """
        根据pw_key获取实际功率值
        :param new_fibre_info: 转换后的光模块数据
        :param pw_key: 功率的key
        :return: pw实际功率值
        """
        pw = None
        for key in pw_key:
            pw = new_fibre_info.get(key)
            if pw:
                break
        return pw

    def trans_dict_key_to_lower(self, fibre_data):
        """
        忽略大小写，去空格匹配
        :param fibre_data: 原始的光模块信息
        :return:new_fibre_info 转换key值后的信息。
        """
        new_fibre_info = {}
        for (key, value) in fibre_data.items():
            new_key = key.replace(" ", "").strip().lower()
            new_fibre_info[new_key] = value
        return new_fibre_info


class Excel(object):
    """
    解析光模块告警门限列表。
    """

    def __init__(self):
        configFileName = ALARM_THRESHOLD_FILE_NAME
        filepath = py_java_env.get("path")
        filename = filepath + os.sep + configFileName
        self.file_name = filename

    def get_excel_to_list(self):
        LOGGER.logInfo("Excel file path:{}".format(self.file_name))
        workbook = xlrd.open_workbook(self.file_name)
        sheet = workbook.sheet_by_name("Sheet1")
        alarm_data = {}
        for index in range(1, sheet.nrows):
            row_data = sheet.row_values(index)

            power_data = {}
            power_data["TxPower"] = int(row_data[3])
            power_data["RxPower"] = int(row_data[4])
            model = row_data[2].strip()
            alarm_data[model] = power_data
        return alarm_data
