# -*- coding: UTF-8 -*-
import cliUtil
import common
import re

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

def execute(cli):
    '''
        本端与远端阵列之间异构LUN链路信息： 
    1 当系统版本为 V300R003C20, V300R003C20SPC200,V300R005C00SPC200, V300R005C00SPC300, V300R005C01, V300R006C00,
      V300R006C00SPC100 之一，且热补丁版本不为V300R003C20SPH203、V300R005C00SPH303、V300R006C00SPH102之一时，进行之后检查；
    2 如回显"FC Link"记录的 Remote Device ID与"ISCSI Link"记录的 Remote Device ID含有相同ID，
                则表示本端与远端设备异构LUN的链路存在FC链路和ISCSI链路混插，巡检不通过。
    '''

    errMsg = ""
    allCliRet = ""
    try:
        #步骤2，判断版本及补丁
        cliRetUpgrade, softwareVersionList, hotPatchVersionList = common.parse_upgradePackage(cli, LANG)
        allCliRet += cliRetUpgrade[1]
        #判断是否含异常场景
        if cliRetUpgrade[0] != True:
            return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "cannot.get.system.info"))

        #检查版本及补丁是否存在于白名单
        flag, errMsg = checkVersionInWhitel(softwareVersionList, hotPatchVersionList, LANG)
        if flag != False:
            return (flag, allCliRet, errMsg)

        #步骤3，获取回显中"FC Link"与"ISCSI Link"的 Remote Device ID
        cmd = "show remote_device elink"
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        allCliRet += cliRet

        #判断是否含异常场景
        if flag != True:
            errMsg = common.getMsg(LANG, "remote.device.idinfomation")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)

        #步骤3 查询远端设备ID是否含有相同
        flag, errMsg = checkRemoteDeviceID(cliRet)

        return (flag, allCliRet, errMsg)
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))

def checkVersionInWhitel(softwareVersionList, hotPatchVersionList, LANG):
    '''
    @summary: 获取所有磁盘信息集合
    @param cli: cli对象
    @param lang: 语言lang
    '''
    softFlag = False
    hotpatchFlag = False

    #系统版本及热补丁白名单    
    softAndHotpatchWhiteMap = {'V300R003C20SPC200':'V300R003C20SPH203',
                                'V300R005C00SPC300':'V300R005C00SPH303',
                                'V300R006C00SPC100':'V300R006C00SPH102',
                                'V300R003C20':'--',
                                'V300R006C00':'--',
                                'V300R006C01':'--',
                                'V300R005C00SPC200':'--',
                                'V300R005C01':'--'}
    #检查系统版本    
    flag, currentSoftVersion, errMsg = common.getCurrentVersion(softwareVersionList, LANG)
    LOGGER.logInfo('currentSoftVersion : %s' % currentSoftVersion)
    if not flag:
        return (cliUtil.RESULT_NOCHECK, errMsg)
    #检查热补丁版本    
    flag, currentHotpatchVersion, errMsg = common.getCurrentVersion(hotPatchVersionList, LANG)
    LOGGER.logInfo('currentHotpatchVersion : %s' % currentHotpatchVersion)
    if not flag:
        return (cliUtil.RESULT_NOCHECK, errMsg)

    #对版本型号及热补丁版本进行适配
    keysList = softAndHotpatchWhiteMap.keys()
    for softversion in keysList:
        if softversion == currentSoftVersion.strip() and '--' != currentSoftVersion.strip():
            softFlag = True
            LOGGER.logInfo("currentSoftVersion is softversion. softversion : %s" % softversion)
            hotpatchVersion = softAndHotpatchWhiteMap.get(softversion)
            if hotpatchVersion != '--' and hotpatchVersion <= currentHotpatchVersion.strip():
                hotpatchFlag = True
                break

    #如果系统版本不涉及则通过                
    if not softFlag:
        return (True, '')
    #如果热补丁版本不匹配则继续步骤3
    if not hotpatchFlag:
        return (False, '')
    else :
        return (True, '')

def checkRemoteDeviceID(cliRet):
    '''
    @summary: 提取出FC于ISCSI的相应回显中的远端设备ID
    @param cliRet: 收集到的远端设备信息回显
    @return: flag , errMsg
    '''
    fcLinkList = []
    iscsiLinkList = []

    fcLinkRemoteID = []
    iscsiLinkRemoteID = []

    #提取出FC及ISCSI的相应回显
    fcLinkIndex = cliRet.find("FC Link")
    iscsiLinkIndex = cliRet.find("ISCSI Link")
    if fcLinkIndex == -1 or iscsiLinkIndex == -1:
        return (cliUtil.RESULT_NOCHECK, common.getMsg(LANG, "remote.device.idinfomation"))

    fcLinkList = cliUtil.getHorizontalNostandardCliRet(cliRet[(fcLinkIndex):iscsiLinkIndex])
    iscsiLinkList = cliUtil.getHorizontalNostandardCliRet(cliRet[(iscsiLinkIndex):])

    #如果在其中之一没有回显，则为正常状态    
    if len(fcLinkList) == 0 or len(iscsiLinkList) == 0 :
        return (True, "")

    fcLinkRemoteID = pickUpRemoteDeviceID(fcLinkList, fcLinkRemoteID)
    LOGGER.logInfo("fcLinkRemoteID : %s" % fcLinkRemoteID)
    iscsiLinkRemoteID = pickUpRemoteDeviceID(iscsiLinkList, iscsiLinkRemoteID)
    LOGGER.logInfo("iscsiLinkRemoteID : %s" % iscsiLinkRemoteID)

    #如果未取到值，则报未检查
    if not fcLinkRemoteID or not iscsiLinkRemoteID:
        return (cliUtil.RESULT_NOCHECK, common.getMsg(LANG, "remote.device.idinfomation"))

    #查看两份ID列表是否含有相同ID    
    setIdenticalID = set(fcLinkRemoteID) & set(iscsiLinkRemoteID)

    #如果含有相同ID 则巡检不通过    
    if setIdenticalID:
        return (False, common.getMsg(LANG, "remote.device.total.idfail", ",".join(setIdenticalID)))
    else:
        return (True, "")

def pickUpRemoteDeviceID(linkMap, linkRemoteID):
    '''
    @summary: 找出回显中的远端设备ID
    @param linkMap: 收集到的远端设备ID 字典列表
    @param linkRemoteID: 提取出的ID 列表
    @return: linkRemoteID
    '''
    #定义关键字常量        
    REMOTEDEV_ID = 'Remote Device ID'

    for link in linkMap:
        remoteDevID = link.get(REMOTEDEV_ID)
        #容错：只有在VALUE有值的情况下，才判断为有效ID          
        if remoteDevID != None:
             linkRemoteID.append(remoteDevID)
    #为提取出的远端设备ID去重
    linkRemoteID = list(set(linkRemoteID))
    return linkRemoteID
