# -*- coding: UTF-8 -*-
from collections import OrderedDict

import cliUtil
import common
import common_utils
from common_cache import get_filesystem_domain_info_cache
from common_cache import is_4u_or_6u_engine_type, is_management_local_port


LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    @summary: 现网局点双活仲裁配置规范检查，如果没有配置第三方仲裁服务器，建议配置；如果配置了第三方仲裁服务器，仲裁服务器到每个控制器至少有一条链路，否则双活在运行过程中有可靠性风险。
    """
    allCliRet = ""
    try:
        #检查双活域是否配置仲裁服务器
        flag, cliRet, errMsg, server_id_list = checkQuorumServer(cli)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag != True:
            return (flag, allCliRet, errMsg)

        # 增加NAS双活检查
        flag, ret, msg, nas_server_id_list = check_nas_quorum_server(cli)
        allCliRet = common.joinLines(allCliRet, ret)
        if flag is not True:
            return flag, allCliRet, msg

        if not server_id_list and not nas_server_id_list:
            return True, allCliRet, ''

        #获取设备控制器ID
        flag, controIdList, errMsg, cliRet= cliUtil.getControllerIdListWithRet(cli, LANG)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag != True:
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)

        server_id_list.extend(nas_server_id_list)
        server_id_set = set(server_id_list)
        err_ser_msg_dict = OrderedDict()
        # 检查每个控制器是否绑定到仲裁服务器上
        flag, cliRet, errMsg, cli_ret_dict_list = checkQuorumServerLink(
            cli, controIdList, server_id_set, err_ser_msg_dict)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag != cliUtil.RESULT_WARNING:
            return flag, allCliRet, errMsg

        # 控制器做进一步引擎检查
        LOGGER.logInfo('updated server id set {}'.format(server_id_set))
        d_flag, mgmt_flag, d_cli_ret, err_msg = logic_engine_info_check(
            cli, cli_ret_dict_list, server_id_set, err_ser_msg_dict)
        all_cli_ret = common.joinLines(allCliRet, d_cli_ret)

        if d_flag:
            # 使用了6U/4U MGMT仲裁
            if err_ser_msg_dict:
                # 建议优化 合并上一步检查项的警告信息
                return cliUtil.RESULT_WARNING, all_cli_ret, ''.join(
                    err_ser_msg_dict.values()) + common.getMsg(LANG, "engine.less.than.two.controllers.suggest")
            return mgmt_flag, all_cli_ret, err_msg

        return flag, all_cli_ret, errMsg

    except common.UnCheckException as e:
        all_cli_ret = common.joinLines(allCliRet, e.cliRet)
        return cliUtil.RESULT_NOCHECK, all_cli_ret, e.errorMsg

    except Exception as exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))


def check_nas_quorum_server(cli):
    """
    检查nas双活仲裁链路信息
    :param cli:
    :return:
    """
    server_id_list = []
    no_quorum_domain_id_list = []
    nas_domain, ret = get_filesystem_domain_info_cache(
        py_java_env, cli, LOGGER
    )

    for nas_domain_info in nas_domain:
        if not common_utils.is_hypermetro_work_mode(nas_domain_info):
            continue
        quorumServerId = nas_domain_info.get("Quorum ID")
        server_id_list.append(quorumServerId)
        if quorumServerId == "" or quorumServerId == "--":
            domain_id = nas_domain_info.get("ID")
            no_quorum_domain_id_list.append(domain_id)

    if len(no_quorum_domain_id_list) != 0:
        msg = common.getMsg(
            LANG, "hyperMetro.domain.not.exit.quorum.server",
            ",".join(no_quorum_domain_id_list)
        )
        return cliUtil.RESULT_WARNING, ret, msg, server_id_list

    return True, ret, '', server_id_list


def checkQuorumServer(cli):
    """
    @summary: 检查每个双活域是否存在仲裁服务器
    @return: (检查结果，CLI回显，错误提示信息）
    """
    cmd = "show hyper_metro_domain general |filterColumn include columnList=Quorum\sServer\sID,ID"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    
    if flag == False: 
        LOGGER.logSysAbnormal()
        errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"双活域","en":"hyper metro domain"}.get(LANG))
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, [])
    
    if flag == cliUtil.RESULT_NOSUPPORT or flag == cliUtil.RESULT_NOCHECK:
        return (cliUtil.RESULT_NOSUPPORT, cliRet, "", [])
    
    errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"双活域","en":"hyper metro domain"}.get(LANG))

    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg, [])
    
    cliRetDictList = cliUtil.getHorizontalCliRet(cliRet)
    #未配置仲裁服务器的双活域
    hyperMetroDomainList = []
    server_id_list = []
    for cliRetDict in cliRetDictList:
        quorumServerId = cliRetDict.get("Quorum Server ID")
        server_id_list.append(quorumServerId)
        if quorumServerId == None:
            return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, server_id_list)
        if quorumServerId == "" or quorumServerId == "--":
            hyperMetroDomainId = cliRetDict.get("ID")
            if hyperMetroDomainId == None:
                return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, server_id_list)
            hyperMetroDomainList.append(hyperMetroDomainId)
            
    if len(hyperMetroDomainList) != 0:
        errMsg = common.getMsg(LANG, "hyperMetro.domain.not.exit.quorum.server", ",".join(hyperMetroDomainList))
        return (cliUtil.RESULT_WARNING, cliRet, errMsg, server_id_list)
    
    return (True, cliRet, "", list(set(server_id_list)))
    

def checkQuorumServerLink(cli, controIdList, server_id_set, err_ser_msg_dict):
    """
    @summary: 检查每个控制器与仲裁服务器之间是否都有仲裁链路
    @return: (检查结果，CLI回显，错误提示信息）
    """
    not_pass_err_msg = []
    cmd = "show quorum_server_link general"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    
    if flag != True: 
        LOGGER.logSysAbnormal()
        errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"阵列与仲裁服务器之间的链路","en":"links between the disk array and quorum servers"}.get(LANG))
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, [])
    
    #不存在仲裁链路检查结果为不通过
    if cliUtil.queryResultWithNoRecord(cliRet):
        errMsg = common.getMsg(LANG, "device.not.exit.quorum.server.link")
        return (False, cliRet, errMsg, [])
    
    cliRetDictList = cliUtil.getHorizontalCliRet(cliRet)
    server_contr_dict = {}
    ret_link_dict_list = []
    for cliRetDict in cliRetDictList:
        server_id = cliRetDict.get("Server Id")
        if server_id not in server_id_set:
            continue
        linkStatus = cliRetDict.get("Link Status")
        localContr = cliRetDict.get("Local Controller")
        if linkStatus == None or localContr == None:
            errMsg = common.getMsg(LANG, "cannot.get.info", {"zh":u"阵列与仲裁服务器之间的链路","en":"links between the disk array and quorum servers"}.get(LANG))
            return (cliUtil.RESULT_NOCHECK, cliRet, errMsg, cliRetDictList)

        if linkStatus == "Link Up":
            server_contr_dict[server_id] = server_contr_dict.get(
                server_id, []) + [localContr]
            ret_link_dict_list.append(cliRetDict)

    for ser_id in server_id_set:
        controller_id_list = server_contr_dict.get(ser_id)
        if not controller_id_list:
            not_pass_controller_list = controIdList
        else:
            not_pass_controller_list = list(set(
                controIdList).difference(set(controller_id_list)))
        if not_pass_controller_list:
            not_pass_controller_list.sort()
            i_msg = common.getMsg(
                LANG, "contor.not.band.quorum.server",
                (ser_id, ",".join(not_pass_controller_list))
            )
            not_pass_err_msg.append(i_msg)
            err_ser_msg_dict[ser_id] = i_msg
            LOGGER.logInfo(not_pass_err_msg)
        elif ser_id in server_id_set:
            # 下一个检查标准不再检查改仲裁服务ID
            server_id_set.remove(ser_id)
            LOGGER.logInfo('remove {} from server id set: {}'.format(
                ser_id, server_id_set))
    if not_pass_err_msg:
        return (cliUtil.RESULT_WARNING, cliRet, "".join(not_pass_err_msg),
                ret_link_dict_list)

    return (True, cliRet, "", ret_link_dict_list)


def logic_engine_info_check(cli, cli_ret_dict_list, server_id_set,
                            err_ser_msg_dict):
    """
    @summary: 检查4u或6u引擎是否有两控仲裁链路
    @return: (是否使用4u或6u管理网卡仲裁，检查结果，CLI回显，错误提示信息）
    """
    cmd = "show enclosure |filterRow column=Logic\sType predict=equal_to value=Engine"  # noqa

    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag is not True:
        LOGGER.logSysAbnormal()
        err_msg = common.getMsg(
            LANG,
            "can.not.get.engine.info"
        )
        return True, cliUtil.RESULT_NOCHECK, cli_ret, err_msg

    enclosure_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
    all_engine_set = set()
    for enclosure in enclosure_dict_list:
        engine_type = enclosure.get('Type')
        all_engine_set.add(enclosure.get('ID'))
        if not is_4u_or_6u_engine_type(engine_type):
            # 非4U/6U引擎 不做处理
            return False, False, cli_ret, err_msg

    mng_loc_p_dict = {}
    for server_link in cli_ret_dict_list:
        port = server_link.get('Local Port')
        ser_id = server_link.get('Server Id')
        if ser_id not in server_id_set:
            continue

        if is_management_local_port(port):
            engine = "{}:{}".format(ser_id, port.split('.')[0])
            mng_loc_p_dict[engine] = mng_loc_p_dict.get(engine, 0) + 1
        elif ser_id in server_id_set:
            # 非管理网口不做判断
            server_id_set.remove(ser_id)
    if not mng_loc_p_dict:
        # 无管理口引擎 不做处理
        return False, False, cli_ret, err_msg

    LOGGER.logError("management local port dict: {}".format(mng_loc_p_dict))

    # 初始化待检查的仲裁服务器对应的引擎
    error_server_engine_dict = {}
    for ser_id in server_id_set:
        err_ser_msg_dict.pop(ser_id, '')
        error_server_engine_dict[ser_id] = list(all_engine_set)
    LOGGER.logInfo(
        "Init error server engine dict: {}".format(error_server_engine_dict))
    for eg, cnt in mng_loc_p_dict.items():
        split_lst = eg.split(':')
        ser_id = split_lst[0]
        engine = ':'.join(split_lst[1:])
        if cnt >= 2 and ser_id in error_server_engine_dict:
            if engine in error_server_engine_dict[ser_id]:
                # 移除通过检查的仲裁服务器对应的引擎
                error_server_engine_dict[ser_id].remove(engine)

    err_msg = ""
    for ser, egs in error_server_engine_dict.items():
        if not egs:
            continue
        i_msg = common.getMsg(
            LANG,
            "engine.less.than.two.controllers",
            (ser, ','.join(egs))
        )
        err_msg += i_msg
        err_ser_msg_dict.pop(ser, '')
        err_ser_msg_dict[ser] = i_msg
    if err_msg:
        LOGGER.logError(
            "error engines of server: {}".format(error_server_engine_dict))

        return True, cliUtil.RESULT_WARNING, cli_ret, err_msg

    return True, True, cli_ret, err_msg
