# -*- coding: UTF-8 -*-
import cliUtil
import common
from event_log_util import fetchLog2Local
from common import UnCheckException
import config
import re
import time

OLD_VER_V3 = 'V300R006C10SPC100'
OLD_VER_V5 = 'V500R007C00SPC100'
OLD_VER_DORADO = 'V300R001C21SPC100'
NEW_VER_V3 = 'V300R006C20'
NEW_VER_V5 = 'V500R007C10'
NEW_VER_DORADO = 'V300R001C30'
LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
ALL_CLI_RET = ''


def execute(cli):
    """
        端口漂移组网检查：
        检查方法(英文)
        步骤1 以admin用户登录设备，并进入developer视图。
        步骤2 查询show logical_port failover_switch service_type=SAN检查故障场景端口漂移开关是否打开。
        步骤3 执行命令show initiator initiator_type=FC查询FC启动器列表。
        步骤4 执行命令show file export_path file_type=running_data导出运行数据，获取HBA info表。
        步骤5 依次查看处于在线状态且已关联主机的FC启动器及WWN，通过启动器WWN从HBA info表中查找启动器对应的主机HBA卡信息，检查HBA卡信息与存储是否兼容（比较“HBA:”与“HN:”之间的信息是否匹配）。
        步骤6 执行命令test logical_port failover service_type=SAN，判断主机和阵列的组网是否满足端口漂移条件。若组网检查不通过，执行show failover_path general命令，查看组网检查不通过原因。

        检查标准
        1.如步骤2中查询到的故障漂移开关为off，则不涉及该项检查。
        2.步骤3中查询到的FC启动器，过滤出运行状态为在线并且已关联主机的FC启动器，如过滤结果为空，则跳转执行步骤6。
        3.如步骤5中FC启动器对应的HBA卡与存储阵列不兼容，则检查不通过。
        4.如步骤6命令的执行结果不为“Command executed successfully.”，则检查不通过。

        修复建议
        1.若步骤5检查不通过，V500R007C00SPC100\V300R006C10SPC100\DoradoV300R001C21SPC100提示信息：主机HBA卡和存储可能存在兼容性风险，请安装V500R007C00SPH102/V300R006C10SPH102/DoradoV300R001C21SPH101或之后补丁，V500R007C10\V300R006C20\DorardoV300R001C30及之后版本提示信息：主机HBA卡和存储可能存在兼容性风险，并建议关闭故障场景端口漂移(开发者视图下执行命令change logical_port failover_switch service_type=SAN switch=off)。
        2.若步骤6检查不通过，提示：端口漂移组网检查不通过，请按照端口漂移标准组网进行组网调整后重试，或关闭故障场景端口漂移开关。检查不通过原因为test logical_port failover service_type=SAN的错误回显。执行"show failover_path general"命令，解析漂移路径状态。

        说明：如果步骤5通过继续检查步骤6，如果步骤5不通过就检查不通过。
    """
    global ALL_CLI_RET
    try:

        currentVersion = str(PY_JAVA_ENV.get("devInfo").getProductVersion())
        deviceType = str(PY_JAVA_ENV.get("devInfo").getDeviceType())

        risk_version_flag, risk_version_msg = checkIsRiskVersion(
            currentVersion, deviceType)
        if not risk_version_flag:
            return cliUtil.RESULT_NOSUPPORT, \
                   common.version_no_support_msg_str(
                       deviceType, currentVersion, "",
                       risk_version_msg, LANG), ''
        
        #步骤2 检查开关
        flag = checkFailoverSwitchOFF(cli)
        # flag为True时表示开关是打开的，其他结果为 no support.
        if flag != True:
            return flag, ALL_CLI_RET, ''
        
        cliRet, __, hotPatchVer = common.getHotPatchVersionAndCurrentVersion(cli, LANG)
        ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
        
        # 步骤3 如果是老版本且打了补丁，则不需要继续做检查。
        if isOldDevVer():
            if currentVersion.startswith("V5") and hotPatchVer >= "V500R007C00SPH102":
                return cliUtil.RESULT_NOSUPPORT, ALL_CLI_RET, ""
            elif deviceType not in config.DORADO_DEVS and currentVersion.startswith(
                    "V300R006") and hotPatchVer >= "V300R006C10SPH102":
                return cliUtil.RESULT_NOSUPPORT, ALL_CLI_RET, ""
            elif deviceType in config.DORADO_DEVS and hotPatchVer >= "V300R001C21SPH101":
                return cliUtil.RESULT_NOSUPPORT, ALL_CLI_RET, ""
        
        cliUtil.enterCliModeFromSomeModel(cli, LANG)

        fcIniList, fcInitiatorDict = getFcInitiator(cli)
        LOGGER.logInfo("fcIniList: %s, " % fcIniList)

        if not fcIniList:
            # 步骤3中查询到的FC启动器，过滤出运行状态为在线并且已关联主机的FC启动器，如过滤结果为空，则跳转执行步骤6。
            flag, errMsg = failOverCliRet(cli)

            # 检查组网
            if flag == False:
                # 不通过，处理错误消息
                errMsg = failedReasonResult(cli)
                return False, ALL_CLI_RET, errMsg

            # 通过
            return flag, ALL_CLI_RET, ''

        __, productVersion, hotPatchVersion, __, __ = common.getProductVersionAndHotPatchVersion(cli, LOGGER, LANG)
        isRisk, needInstallHotpatchVersion = isRiskVersionFcCauseRisk(cli, productVersion, hotPatchVersion)
        if isRisk:
            LOGGER.logInfo(
                "hot patch version is lower than requst, and fc ini is more than 80 or controller is not normal.")
            return cliUtil.RESULT_NOCHECK, ALL_CLI_RET, common.getMsg(LANG, "install.fc.driver.risk.patch", needInstallHotpatchVersion)

        # 根据启动器wwn从config文件中获取 hba信息。
        hbaInfoDict = getHBAInfoFromConfig(cli, fcIniList)
        LOGGER.logInfo("getHBAInfoFromConfig hbaInfoDict: %s, " % hbaInfoDict)

        # 检查HBA卡是否在白名单中
        notInWhiteList = checkWhiteHBAInfo(hbaInfoDict)
        LOGGER.logInfo("checkWhiteHBAInfo notInWhiteList: %s, " % notInWhiteList)

        # 如果存在HBA不在白名单中，返回不通过
        if notInWhiteList:
            # 老版本提示
            if isOldDevVer():
                LOGGER.logInfo("checkWhiteHBAInfo isOldDevVer hotPatchVer: %s" % hotPatchVer)
                return False, ALL_CLI_RET, common.getMsg(LANG, "cli_software_checkFailover.step5.fail.oldver")
            # 新版本提示
            else:
                LOGGER.logInfo("checkWhiteHBAInfo isOldDevVer new version.")
                return False, ALL_CLI_RET, common.getMsg(LANG, "cli_software_checkFailover.step5.fail.newver")

        # 检查组网
        flag, errMsg = failOverCliRet(cli)
        if flag == False:
            errMsg = failedReasonResult(cli)
            return False, ALL_CLI_RET, errMsg

        return flag, ALL_CLI_RET, errMsg

    except UnCheckException, unCheckException:
        LOGGER.logInfo(u"UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        if not unCheckException.flag:
            return cliUtil.RESULT_NOCHECK, unCheckException.cliRet, unCheckException.errorMsg
        return unCheckException.flag, unCheckException.cliRet, unCheckException.errorMsg
    except Exception, exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, ALL_CLI_RET, common.getMsg(LANG, "query.result.abnormal")

    finally:
        cliUtil.enterCliModeFromSomeModel(cli, LANG)


def checkIsRiskVersion(currentVersion, productModel):
    """
    检查版本是否是风险版本
    :param currentVersion:
    :param productModel:
    :return:
    """
    risk_map = {"V5": "V500R007C00SPC100", "V3": "V300R006C10SPC100",
                "dorado": "V300R001C21"}
    risk_version = risk_map.get(
        common.get_product_key(productModel, currentVersion), "")
    if not risk_version:
        return False, ""
    if currentVersion >= risk_version:
        return True, ""
    return False, common.getMsg(LANG,
                                'msg.risky.version.and.later',
                                risk_version)


def checkWhiteHBAInfo(hbaInfoDict):
    """
    检查每个启动器的HBA卡信息是否在白名单中
    :param hbaInfoDict:
    :return:
    """
    notInWhiteList = []
    deviceType = str(PY_JAVA_ENV.get("devInfo").getDeviceType())
    model = "DORADO" if deviceType in config.DORADO_DEVS else "VSERIES"
    currentVersion = str(PY_JAVA_ENV.get("devInfo").getProductVersion())
    arrayVersion = currentVersion[0:11]
    
    # 其他在研版本的，都按照最近的有的版本检查。
    if not arrayVersion in config.CHECK_FAILOVER_RISK_VERSION.get(model, []):
        if model == "DORADO" and arrayVersion > "V300R002C00":
            LOGGER.logInfo("DORADO arrayVersion:%s is new version, need check use: V300R002C00" % arrayVersion)
        elif model!= "DORADO" and arrayVersion[0:8] == "V300R006" and arrayVersion > "V300R006C21":
            LOGGER.logInfo("NOT DORADO arrayVersion:%s is new version, need check use: V300R006C21" % arrayVersion)
        elif model!= "DORADO" and arrayVersion[0:8] == "V500R007" and arrayVersion > "V500R007C20":
            LOGGER.logInfo("NOT DORADO arrayVersion:%s is new version, need check use: V500R007C20" % arrayVersion)
        else:
            return notInWhiteList

    for wwpn in hbaInfoDict:
        hbaInfo = hbaInfoDict.get(wwpn)
        findFlag = False
        for hbaStr in config.HBA_WHITE_LIST:
            if hbaStr.startswith(hbaInfo):
                findFlag = True
                break

        if not findFlag:
            notInWhiteList.append(wwpn)

    return notInWhiteList


def getHBAInfoFromConfig(cli, fcIniList):
    """
    从config文件中获取HBA信息
    :param cli:
    :param fcIniList:
    :return:
    """
    global ALL_CLI_RET
    hbaInfoDict = {}
    cmd = "show file export_path file_type=running_data"
    deleteCmd = "delete file filetype=running_data"
    flag, localPath, cliRet, errMsg = fetchLog2Local(cmd, deleteCmd, PY_JAVA_ENV, cli, LOGGER, LANG)
    LOGGER.logInfo("fetchLog2Local localPath: %s, " % localPath)

    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag != True:
        for i in range(3):
            flag, localPath, cliRet, errMsg = fetchLog2Local(cmd, deleteCmd, PY_JAVA_ENV, cli, LOGGER, LANG)
            ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
            if flag:
                break
            time.sleep(30)

    if not localPath:
        return hbaInfoDict

    lines = []
    try:
        f = open(localPath, "r")
        lines = f.readlines()
        f.close()
    except:
        LOGGER.logInfo("fetchLog2Local open localPath: %s error. " % localPath)

    totalLine = len(lines)
    for i in range(totalLine):
        line = lines[i].strip()
        if "WWPN" in line and (i + 1) < totalLine and "HBA:" in lines[i + 1]:
            wwpn = line.split(":")[-1].strip()
            if not wwpn:
                continue
            if wwpn not in fcIniList:
                continue

            tmpLine = lines[i + 1].strip()
            res = re.search("HBA:(.*)", tmpLine)
            if res:
                hbaInfo = res.group(1).split("HN:")[0].strip()
                if not hbaInfo:
                    continue

                LOGGER.logInfo("HBA info: wwpn:[%s], hba:[%s]" % (wwpn, hbaInfo))
                hbaInfoDict[wwpn] = "HBA:" + hbaInfo

    return hbaInfoDict


def isRiskVersionFcCauseRisk(cli, currentSoftVersion, currentHotpatchVersion):
    '''
    @summary: 工具规避在收集日志时触发的问题
    @param softwareVersionList: 当前阵列的版本所有控制器的版本列表
    @param hotPatchVersionList: 当前阵列的版本所有控制器的补丁版本列表
    retrun：False：表示是风险版本
            True：表示不是风险版本
    '''

    # 系统版本及热补丁风险版本
    riskVersionDict = {
        'V300R006C20': 'V300R006C20SPH015',
        'V300R006C20SPC100': 'V300R006C20SPH115',
        'V500R007C10': 'V500R007C10SPH015',
        'V500R007C30SPC100': 'V500R007C30SPH105',
        'V300R001C21SPC100': 'V300R001C21SPH112',
        'V500R007C10SPC100': 'V500R007C10SPH115',
        'V300R002C10SPC100': 'V300R002C10SPH101',
        'V300R006C50SPC100': 'V300R006C50SPH105',
        'V300R001C30SPC100': 'V300R001C30SPH106'
    }

    # 对版本型号及热补丁版本进行适配,如果系统版本不涉及则通过
    if currentSoftVersion not in riskVersionDict:
        return False, ''

    needInstallHotpatchVersion = riskVersionDict.get(currentSoftVersion)
    # 如果热补丁版本匹配则通过
    if currentHotpatchVersion != '--' and currentHotpatchVersion >= needInstallHotpatchVersion:
        return False, ''

    # 如果热补丁版本不匹配
    # 需判断启动器数量和控制器冗余
    cmd = "show initiator initiator_type=FC"
    cliRet = cli.execCmdNoLogTimout(cmd, 10*60)
    iniDictList = cliUtil.getHorizontalCliRet(cliRet)
    normalInitiatorCount = 0
    for iniDict in iniDictList:
        if iniDict.get("Running Status", '') == "Online":
            normalInitiatorCount += 1

    isAllCtrlNormal = True
    cmd = "show controller general |filterColumn include columnList=Health\sStatus,Running\sStatus,Controller"
    cliRet = cli.execCmdNoLogTimout(cmd, 10*60)
    controllerDictList = cliUtil.getVerticalCliRet(cliRet)
    for controllerDict in controllerDictList:
        if controllerDict.get("Health Status", '') != "Normal":
            isAllCtrlNormal = False
            break

        if controllerDict.get("Running Status", '') != "Online":
            isAllCtrlNormal = False
            break

    LOGGER.logInfo("hot patch version is lower than request, normal fc Initiator Count:%s, isAllCtrlNormal:%s" % (
                  normalInitiatorCount, isAllCtrlNormal))

    if normalInitiatorCount <= 200 and isAllCtrlNormal:
        return False, ''

    # 未安装对应补丁，且启动器数量超过80个或控制器不正常，限制收集。
    return True, needInstallHotpatchVersion


def getFcInitiator(cli):
    """
    获取FC启动器信息，在线的，配置了主机的。
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    cmd = "show initiator initiator_type=FC"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag != True:
        raise UnCheckException(errMsg, ALL_CLI_RET)

    fcInitiatorDict = {}
    tmpResDictList = cliUtil.getHorizontalCliRet(cliRet)
    fcInitiatorList = []
    for resDict in tmpResDictList:
        wwn = resDict.get("WWN", '')
        runningStatus = resDict.get("Running Status", '')
        hostId = resDict.get("Host ID", '')

        if wwn and runningStatus == 'Online' and hostId != '--':
            fcInitiatorList.append(wwn)
            fcInitiatorDict[wwn] = hostId

    return fcInitiatorList, fcInitiatorDict


def checkFailoverSwitchOFF(cli):
    """
    检查是否端口漂移开关是否打开
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    cmd = "show logical_port failover_switch service_type=SAN"
    __, cliRet, __ = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)

    if "Failover Switch : On" in cliRet:
        return True

    return cliUtil.RESULT_NOSUPPORT


def failOverCliRet(cli):
    """
    测试组网
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    # 组网检测
    cmd = "test logical_port failover service_type=SAN"
    LOGGER.logExecCmd(cmd)
    flag, cliRet, errMsg = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)

    if cliUtil.queryResultWithNoRecord(cliRet):
        return True, ''

    # 命令不存在场景时，返回不涉及
    if not cliUtil.hasCliExecPrivilege(cliRet):
        return cliUtil.RESULT_NOSUPPORT, errMsg

    return False, common.getMsg(LANG, "check.failover.notpass")


def failedReasonResult(cli):
    """
    根据查询结果，处理组网错误提示。
    :param cli:
    :return:
    """
    global ALL_CLI_RET
    cmd = "show failover_path general"
    flag, cliRet, errorMsg = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, LANG)
    ALL_CLI_RET = common.joinLines(ALL_CLI_RET, cliRet)
    if flag != True:
        raise UnCheckException(common.getMsg(LANG, "check.failover.notpass"), ALL_CLI_RET)

    suggessionList = []
    allSuggession = common.getMsg(LANG, "cli_software_checkFailover.common")
    cliRetList = cliUtil.getHorizontalCliRet(cliRet)
    trueResult = ["True", "NA"]
    for oneRet in cliRetList:
        if oneRet.get("Available") == "True":
            continue
        LOGGER.logInfo(oneRet)
        suggession = ''
        if oneRet.get("Target Port") == "--":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.0.notpass")
            suggessionList.append(
                '\n' + str(len(suggessionList) + 1) + ') ' + common.getMsg(LANG, "cli_software_checkFailover.common.1",
                                                                           (oneRet.get("Source Port"),
                                                                            oneRet.get("Target Port")))
                + suggession)
            continue

        sourcePortNPIVResult = oneRet.get("Source Port NPIV")
        if sourcePortNPIVResult == "False":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.1.notpass")
        elif sourcePortNPIVResult == "--":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.1.notcheck")

        sourceSwitchNPIVResult = oneRet.get("Source Switch NPIV")
        if sourceSwitchNPIVResult == "False":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.2.notpass")
        elif sourceSwitchNPIVResult == "--":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.2.notcheck")

        targetPortNPIVResult = oneRet.get("Target Port NPIV")
        if targetPortNPIVResult == "False":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.3.notpass")
        elif targetPortNPIVResult == "--":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.3.notcheck")

        targetSwitchNPIVResult = oneRet.get("Target Switch NPIV")
        if targetSwitchNPIVResult == "False":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.4.notpass")
        elif targetSwitchNPIVResult == "--":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.4.notcheck")

        if oneRet.get("Free Link Enough") not in trueResult:
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.5.notpass")

        if oneRet.get("Redundancy") != "True":
            suggession += "\n" + common.getMsg(LANG, "cli_software_checkFailover.6.notpass")

        if oneRet.get("Failover Status") == "Taking over":
            suggession += "\n" + (common.getMsg(LANG, "cli_software_checkFailover.7.notpass"))

        if oneRet.get("Failover Status") == "Failed-over":
            suggession += "\n" + (common.getMsg(LANG, "cli_software_checkFailover.8.notpass"))

        if suggession:
            LOGGER.logInfo(suggession)
            suggessionList.append(
                '\n' + str(len(suggessionList) + 1) + ') ' + common.getMsg(LANG, "cli_software_checkFailover.common.1",
                                                                           (oneRet.get("Source Port"),
                                                                            oneRet.get("Target Port")))
                + suggession)

    if suggessionList:
        return allSuggession + ''.join(suggessionList)

    suggession = common.getMsg(LANG, "check.failover.notpass")
    return suggession


def isOldDevVer():
    """
    判断是否是老版本，做不通过处理。
    :return:
    """
    deviceType = str(PY_JAVA_ENV.get("devInfo").getDeviceType())
    currentVersion = str(PY_JAVA_ENV.get("devInfo").getProductVersion())
    if deviceType in config.DORADO_DEVS:
        if currentVersion < NEW_VER_DORADO:
            return True
    else:
        if (currentVersion.startswith("V3") and currentVersion < NEW_VER_V3) or \
                (currentVersion.startswith("V5") and currentVersion < NEW_VER_V5):
            return True

    return False
