# -*- coding:utf-8 -*-
import re

import cliUtil
import common
import traceback
from common import UnCheckException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
SYSTEM_SOFTWARE_VERSION = "V500R007C60SPC300"

NO_PROBLEM_PATCH = 302


def execute(cli):
    dm_check_lif_on_bond = DMCheckLifOnBond(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = dm_check_lif_on_bond.execute_check()
    return flag, "\n".join(dm_check_lif_on_bond.all_ret_list), msg


class DMCheckLifOnBond:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []

    def execute_check(self):
        """
        步骤1 以admin用户登录设备；
        步骤2 执行命令：show upgrade package，获取系统软件版本和热补丁版本信息；
        步骤3 执行命令：show logical_port general|
        filterColumn include columnList=Logical\\sPort\\sName,Home\\sPort\\
        sType,Failover\\sGroup\\sID，查询逻辑端口的详细信息。
        检查标准：
        1 若步骤2中系统软件版本不为V500R007C60SPC300 Kunpeng，则检查结果为通过，否则继续检查；
        2 若步骤2中系统软件版本为V500R007C60SPC300 Kunpeng，且安装的补丁版本为SPH302或更新版本，则检查结果为通过，否则继续检查；
        3 若步骤3中存在Home Port Type字段值为Bond或VLAN的逻辑端口，且Failover Group ID字段值不在[1,1024]范围内的逻辑端口，则检查结果为不通过，否则检查结果为通过。
        :param self:
        :return:
        """
        try:
            # 查询到的版本不在名单内检查项直接通过
            if not self.is_risk_version():
                return True, ''
            flag, info_content, _ = self.query_logical_port_info()
            # 对初步处理后的回显再处理
            not_pass_port_name_list = self.filter_info(info_content)
            # 没有赛选出不符合要求的端口则检查项通过
            if not not_pass_port_name_list:
                return True, ''
            err_message = []
            for i in not_pass_port_name_list:
                logic_port_name = i.get('Logical Port Name')
                logic_id = i.get('Failover Group ID')
                err_message.append(common.getMsg(
                    self.lang, "check.lif.on.bond.err_port",
                    (logic_port_name, logic_id)))
            return (False, "".join(err_message))
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def is_risk_version(self):
        """
        判断产品是否满足检查的版本及型号范围
        :param self:
        :return: True 是风险版本， False 非风险版本
        """
        flag, cli_ret, err_msg, software_version, patch = common.getVersion(
            self.cli, self.lang)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        if software_version == SYSTEM_SOFTWARE_VERSION and not self.is_installed_patch(patch):
            return True
        return False

    @staticmethod
    def is_installed_patch(patch):
        find_patch = re.findall(r"SPH(\d+)", patch)
        if not find_patch:
            return False
        return int(find_patch[0]) >= NO_PROBLEM_PATCH

    def query_logical_port_info(self):
        """
        查询所有的逻辑端口信息
        :param self:
        :return:
        """
        cmd = "show logical_port general|" \
              "filterColumn include columnList=" \
              "Logical\\sPort\\sName,Home\\sPort\\sType,Failover\\sGroup\\sID"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True,
                                                    self.lang)
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        # 按逐行字典的方式获取水平表格形式的cli回显列表
        logical_port_info_list = cliUtil.getHorizontalCliRet(ret)

        return (True, logical_port_info_list, "")

    @staticmethod
    def filter_info(content):
        """
        过滤掉Home Port Type非Bond和VLAN且Failover Group ID不在范围
        [1,1024]的字段
        :param content
        :return: [{}.....]
        """
        # 过滤掉Home Port Type非Bond和VLAN的字段
        temp_object = list(filter(
            lambda x: x.get('Home Port Type') in ("Bond", "VLAN"), content))
        # 收集 Failover Group ID不在[1,1024]的端口名字。
        target = []
        for i in temp_object:
            if i.get('Failover Group ID').isdigit() and 1 <= int(
                    i.get('Failover Group ID')) <= 1024:
                continue
            target.append(i)
        return target
