# -*- coding: UTF-8 -*-
import common
import cliUtil
import re
import traceback
from common_utils import get_err_msg
from common import UnCheckException
from risk_version_config import PROTECT_CAPACITY_RISK_VERSION

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env

COMPARE_VERSION = "V300R002C10SPC100"


def execute(cli):
    """
    检查HyperCDP保护容量
    :param cli:
    :return:link_redundant
    """
    protect_capacity = ProtectCapacityCheck(cli)
    flag, msg = protect_capacity.execute_check()
    return flag, "\n".join(protect_capacity.all_cli_ret), msg


class ProtectCapacityCheck:
    def __init__(self, cli):
        self.cli = cli
        self.lang = LANG
        self.env = PY_JAVA_ENV
        self.logger = LOGGER
        self.all_cli_ret = []

    def execute_check(self):

        try:
            return self.check_protect_capacity()
        except UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            err_msg = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(self.lang, err_msg)

    def check_protect_capacity(self):
        """
        检查版本和已创建hyperCDP一致性组的数量（CDP CG obj count）
        :return:True or False 通过或者不通过， str 错误提示
        """
        is_passed, soft_version, pat_version, risk_pat_version =\
            self.is_pass_version()
        if is_passed:
            return True, ""
        if self.is_cdp_cg_count_equal_zero():
            return True, ""
        if soft_version == COMPARE_VERSION:
            err_msg = "software.hypercdp.protect.capacity.not.pass"
            param = (soft_version, pat_version, risk_pat_version)
            return False, get_err_msg(self.lang, err_msg, param)
        else:
            err_msg = "software.hypercdp.protect.capacity.no.patch.version"
            return False, get_err_msg(self.lang, err_msg)

    def is_pass_version(self):
        """
        是否是通过的版本
        :return:True or False 是或者不是，软件版本，补丁版本，风险补丁版本呢
        """
        flag, cli_ret, err_msg, soft_version, pat_version = common.getVersion(
            self.cli, self.lang)
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        risk_pat_version = PROTECT_CAPACITY_RISK_VERSION.get(soft_version)
        if risk_pat_version != "--" and risk_pat_version <= pat_version:
            return True, "", "", ""
        return False, soft_version, pat_version, risk_pat_version

    def is_cdp_cg_count_equal_zero(self):
        """
        hyperCDP一致性组的数量是否为0
        :return:True or False 是或者不是
        """
        cmd = "cdp showCgIdBitmap 0"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInDebugModel(
            self.cli, cmd, self.lang)
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        cliUtil.enterCliModeFromDebugModel(self.cli, self.lang)
        cdp_cg_count = re.search(r"CDP CG obj count\s*:\s*(\d+).",
                                 cli_ret).group(1)
        return int(cdp_cg_count) == 0
