# -*- coding: UTF-8 -*-
import common
import cliUtil

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

BUG_DEVICE_MODEL_MSG_LIST = ['S2600T', 'S5300T', 'S5500T', 'S5600T',
                             'S5800T', 'S6800T', '18500 V1', '18800 V1',
                             '18800F V1', 'HVS85T', 'HVS88T', '2100 V3',
                             '2200 V3', '2600 V3', '2800 V3', '5300 V3',
                             '5500 V3', '5600 V3', '5800 V3', '6800 V3',
                             '18500 V3', '18800 V3']
BUG_DEVICE_VERSION_MSG_LIST = ['T V200R002C20', 'T V200R002C20SPC100',
                               'T V200R002C20SPC200', 'T V200R002C30',
                               'T V200R002C30SPC100', '18000 V100R001C20',
                               '18000 V100R001C20SPC100',
                               '18000 V100R001C20SPC200',
                               '18000 V100R001C30',
                               '18000 V100R001C30SPC100',
                               'V300R001C00SPC200', 'V300R001C10',
                               'V300R001C10SPC100', 'V300R001C10SPC200',
                               'V300R001C20', 'V300R001C20SPC100',
                               'V300R002C10', 'V300R002C10SPC100',
                               'V300R002C10SPC200', 'V300R003C00',
                               'V300R003C00SPC100']


def execute(cli):
    '''
    @summary: 检查当前设备型号和版本号，且是否包含4K盘
    '''
    cli_ret = ""
    err_msg = ""
    flag = True
    try:
        #查询产品型号
        result, product_model, cli_ret, err_msg = \
            cliUtil.getProductModelWithCliRet(cli, LANG)

        if result is not True:
            return result, cli_ret, err_msg
        #获取版本号和补丁版本号
        resultList, currentVersionDictList, hotPatchVersionDictList = common.parse_upgradePackage(cli, LANG)
        cli_ret = common.joinLines(cli_ret, resultList[1])
        if resultList[0] is not True:
            return cliUtil.RESULT_NOCHECK, cli_ret, resultList[2]
        
        result, product_version, errInfo = \
            common.getCurrentVersionFilterBVer(currentVersionDictList, LANG)
        if not result:
            err_msg += errInfo
            return cliUtil.RESULT_NOCHECK, cli_ret, err_msg
        # 检查设备是否需要进行非4KLUN对齐检查
        is_bug_device = check_bug_device(product_model, product_version)
        if not is_bug_device:
            info = common.version_no_support_msg(
                product_model,
                product_version,
                BUG_DEVICE_MODEL_MSG_LIST,
                BUG_DEVICE_VERSION_MSG_LIST,
                LANG)
            cli_ret = common.joinLines(cli_ret, info)
            return cliUtil.RESULT_NOSUPPORT, cli_ret, info
        # 检查设备硬盘扇区大小
        disk_sector_ret, check_lun_flag = check_disk_sector_size(cli)
        cli_ret = common.joinLines(cli_ret, disk_sector_ret[1])
        if disk_sector_ret[0] is not True:
            return disk_sector_ret[0], cli_ret, disk_sector_ret[2]
        if not check_lun_flag:
            return True, cli_ret, ''
        #获取当前cli显示模式
        checkRet, cmdRet, errInfo = getCliCapacityMode(cli, LANG)
        cli_ret += "\n" + cmdRet
        if checkRet != True:
            return checkRet, cli_ret, errInfo
        
        cliConfigInfoDictlist = cliUtil.getVerticalCliRet(cmdRet)
        capacityMode = ""
        for cliConfigInfoDict in cliConfigInfoDictlist:
            if "Capacity Mode" in cliConfigInfoDict:
                capacityMode = cliConfigInfoDict["Capacity Mode"]
        
        #无法获取CLI显示模式
        if capacityMode == "":
            err_msg = common.getMsg(LANG, "cannot.get.cli.config.info")
            return cliUtil.RESULT_NOCHECK, cli_ret, err_msg
        
        #根据当前模式确定是否需要切换模式
        if capacityMode.lower() != "precise":
            #切换后执行show lun general得到的所有lun的容量单位均为B
            checkRet, cmdRet, errInfo = changeCapacityMode(cli, "precise", LANG)
            cli_ret += "\n" + cmdRet
            if checkRet != True:
                return checkRet, cli_ret, errInfo
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cli_ret,
                common.getMsg(LANG, "query.result.abnormal"))
    
    try:
        #获取lun信息
        checkRet, cmdRet, errInfo = getLunInfo(cli, LANG)
        cli_ret += "\n" + cmdRet
        if checkRet != True:
            return checkRet, cli_ret, errInfo
        
        #检查是否存在lun,不存在lun则不存在问题
        if cliUtil.queryResultWithNoRecord(cmdRet):
            return True, cli_ret, err_msg
        
        # 获取每个lun的容量
        cliRetLinesList = get_lun_capacity(cmdRet)
        if len(cliRetLinesList) == 0:
            err_msg = common.getMsg(LANG, "cannot.get.lun.info")
            LOGGER.logNoPass("Cannot get information about LUN")
            return cliUtil.RESULT_NOCHECK, cli_ret, err_msg
        
        #检查所有lun是否是4k对齐
        notEdevlunIdList = []
        not4kAlignLunIdList, problemLunIdList = checkLunIs4kAlign(cliRetLinesList)
        if len(not4kAlignLunIdList) == 0 and len(problemLunIdList) == 0:
            return True, cli_ret, err_msg
        
        notEdevlunIdList += problemLunIdList
        
        if len(not4kAlignLunIdList) >= 300:
            common.refreshProcess(py_java_env, 20, LOGGER)
        
        #检查非4K对齐的lun，是否不为edevlun
        checkRet, problemLunIdList, getInfoFailedLunIdList, cmdRet= checkLunAllIsEdevlun(cli, not4kAlignLunIdList, LANG)
        cli_ret += "\n" + cmdRet
        notEdevlunIdList += problemLunIdList
         
        if len(notEdevlunIdList) != 0:
            flag = False
            err_msg += \
                common.getMsg(LANG, "not.align.4k.and.is.edevlun",
                              ','.join(notEdevlun[0] for notEdevlun in
                                       notEdevlunIdList))
        if len(getInfoFailedLunIdList) != 0:
            flag = False
            err_msg += common.getMsg(LANG, "cannot.get.lun.disk.location",
                                     ', '.join(getInfoFailedLunIdList))
    
        return flag, cli_ret, err_msg
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cli_ret,
                common.getMsg(LANG, "query.result.abnormal"))
    
    finally:
        #切换cli显示模式到原模式
        changeCapacityMode(cli, capacityMode.lower(), LANG)


def check_bug_device(product_model, product_version):
    """
    1 如果步骤2中设备型态是T系列且系统软件版本大于等于V200R002C30SPC200，
    则检查结果为不涉及，否则继续检查；
    2 如果步骤2中设备型态是18000系列且系统软件版本大于等于V100R001C30SPC200，
    则检查结果为不涉及，否则继续检查；
    3 如果步骤2中设备型态是V3系列且系统软件版本等于V300R001C20SPC200，
    或大于等于V300R003C10
    则检查结果为不涉及，否则继续检查；
    :param product_model:
    :param product_version:
    :return:
    """
    product_model_upper = product_model.upper()
    product_version_upper = product_version.upper()
    # dorado 不涉及
    if "DORADO" in product_model_upper:
        return False
    # 18000 V1系列
    if product_version_upper.startswith("V1") and \
            (product_model_upper.startswith("18") or
             product_model_upper.startswith("HVS")):
        # 本检查项涉及C20-C30
        if product_version_upper < "V100R001C20" \
                or product_version_upper >= "V100R001C30SPC200":
            return False
        else:
            return True
    # T系列
    if product_model_upper.startswith("S") \
            and product_model_upper.endswith("T"):
        if product_version_upper >= "V200R002C30SPC200":
            return False
        else:
            return True
    # V3系列
    if "V3" in product_model_upper:
        if product_version_upper == "V300R001C20SPC200" or \
                product_version_upper >= "V300R003C10":
            return False
        else:
            return True
    # 其他系列不涉及
    return False


def check_disk_sector_size(cli):
    """
    获取硬盘扇区大小信息，查看是否存在大于等于4KB，小于5KB扇区大小的盘
    :param cli:
    :return: cli_ret,是否继续检查LUN
    """
    cmd = 'show disk general |filterColumn ' \
          'include columnList=ID,Sector\\sSize'
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True,
                                                        LANG)
    if flag is not True:
        return (flag, cli_ret, err_msg), False
    disk_id_sector_size_list = cliUtil.getHorizontalCliRet(cli_ret)
    for id_sector_dict in disk_id_sector_size_list:
        sector_size = id_sector_dict.get('Sector Size', '0B')
        sector_kb_size = common.changUnit2KB(sector_size)
        if 4 <= sector_kb_size[1] < 5:
            return (flag, cli_ret, err_msg), True
    return (flag, cli_ret, err_msg), False


def getCliCapacityMode(cli, lang):
    """
    @summary: 获取cli显示模式
    @param cli: cli链接
    @return: cmdRet 命令回文
    """    
    cmd = "show cli configuration"
    cmdRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    return cmdRet
    
def changeCapacityMode(cli, mode, lang):
    """
    @summary: 切换cli显示模式
    @param cli: 
            lang:
    @return: cmdRet 命令回显
    """
    cmd = "change cli capacity_mode=%s" % mode
    cmdRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    return cmdRet
    
def getLunInfo(cli, lang):
    """
    @summary: 获取lun信息
    @param cli: 
            lang:
    @return: cmdRet 命令回显
    """
    cmd = "show lun general |filterRow column=Disk\\sLocation " \
          "predict=equal_to value=Internal|filterColumn " \
          "include columnList=ID,Capacity"
    cmdRet = cliUtil.excuteCmdTimeOutInCliMode(cli, cmd, True, lang, cliUtil.TIME_OUT_BATCH_CLI)
    return cmdRet

def checkLunIs4kAlign(cliRetLinesList):
    """
    @summary: 检查lun的容量是否4K对齐
    @param cliRetLinesList: lun信息结果解析的字典列表,且所有lun的容量单位必须为B
    @return: not4kAlignLunIdList 非4k对齐的lun ID列表(包含容量显示不全的ID)
            problemLunIdList 问题lun ID列表
    """
    not4kAlignLunIdList = []
    problemLunIdList = []
    for lunInfoDict in cliRetLinesList:
        lunId = lunInfoDict["ID"]
        lunCapacity = lunInfoDict["Capacity"]

        #容量显示不全，记录其ID，在后续一并判断容量和是否为Edevlun
        if '..' in lunCapacity:
            not4kAlignLunIdList.append(lunId)
            continue
        
        lunCapacityValue = lunCapacity.replace("B", "")
        lunCapacityNum = int(lunCapacityValue)
        if lunCapacityNum % (4 * 1024) != 0:
            #扩容以block为单位，需要计算扩到4KB整数倍需要多少个Block。 1Block = 512B, lun容量一定是512B的倍数
            expansionBlock = str((4096 - lunCapacityNum % 4096) / 512)
            problemLun = (lunId.ljust(15), expansionBlock)
            problemLunIdList.append(problemLun)
            
    return not4kAlignLunIdList, problemLunIdList
    
def checkLunAllIsEdevlun(cli, lunIdList, lang):
    """
    @summary: 检查lun是否全部为edevlun
    @param lunIdList: 需要检查的LUN
    @return:True/False: 存在lun为非edevlun 
            notEdevlunIdList： 非edevlun的ID列表
            getInfoFailedLunIdList:无法获取详细信息或Disk Loaction的lun
            cliRet：所有lun详细信息
    """
    flag = True
    cliRet = ""
    notEdevlunIdList = []
    getInfoFailedLunIdList = []
    
    lunNum = len(lunIdList)
    if lunNum >= 300:
        currentProcess = 20
        stepProcess = (1.0 / lunNum) * (100 - currentProcess)
    
    for lunId in lunIdList:
        cmd = "show lun general lun_id=%s" % lunId
        cmdRet, cmdInfo, errInfo = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        
        if lunNum >= 300:
            currentProcess = common.refreshProcessByStep(currentProcess, stepProcess, py_java_env, LOGGER)
        
        #无法获取原始信息的lun做记录上报
        if cmdRet != True:
            flag = False
            getInfoFailedLunIdList.append(lunId)
            continue
        
        #获取capacity和Disk Location,并判断是否为edevlun
        checkRet, capacity, diskLocation = getLunDiskCapacityAndLocation(cmdInfo)
        
        #无法获取Disk Location的lun也记录上报
        if not checkRet:
            flag = False
            getInfoFailedLunIdList.append(lunId)
            continue
        
        #如果容量为4K整数倍，则不存在问题
        lunCapacityValue = capacity.replace("B", "")
        lunCapacityNum = int(lunCapacityValue)
        if lunCapacityNum % (4 * 1024) == 0:
            continue
        
        #如果不是edevlun，则记录上报
        if diskLocation != "External":
            flag = False
            #扩容以block为单位，需要计算扩到4KB整数倍需要多少个Block。 1Block = 512B, lun容量一定是512B的倍数
            expansionBlock = str((4096 - lunCapacityNum % 4096) / 512)
            problemLun = (lunId.ljust(15), expansionBlock)
            notEdevlunIdList.append(problemLun)
            #存在问题的lun才将回显传回
            cliRet += cmdInfo
    
    return (flag, notEdevlunIdList, getInfoFailedLunIdList, cliRet)
      
def getLunDiskCapacityAndLocation(cmdRet):
    """
    @summary: 根据LUN信息获取Disk Location字段值和容量
    @param cmdRet: 单个lun的信息
    @return: （True/False, capacity, diskLocation）
    """ 
    lunInfoDictList = cliUtil.getVerticalCliRet(cmdRet)    
    for lunInfoDict in lunInfoDictList:
        if "Disk Location" in lunInfoDict and "Capacity" in lunInfoDict:
            capacity = lunInfoDict["Capacity"] 
            diskLocation = lunInfoDict["Disk Location"] 
            return (True, capacity, diskLocation)
        
    return (False, "", "")


def get_lun_capacity(cmdRet):
    """
    @summary: 解析show lun general |filterColumn include columnList=ID,Capacity回文，得到每个lun的容量值
    @param cmdRet:show lun general |filterColumn include columnList=ID,Capacity回文 
    @return: idAndCapacityDictList:ID为key，Capacity为值的字典列表
    """
    
    idAndCapacityDictList=[]
    for line in cmdRet.splitlines():
        idAndCapacityDict={}
        #cli结束
        if cliUtil.CLI_RET_END_FLAG in line:
            break
        
        #非数据信息
        lineInfo = line.split()
        if len(lineInfo) != 2:
            continue
        
        id = lineInfo[0].strip()
        capacity = lineInfo[1].strip()

        if id.upper() == "ID":
            continue
        if id.startswith("-"):
            continue
        
        #添加lun信息
        idAndCapacityDict["ID"] = id
        idAndCapacityDict["Capacity"] = capacity
        idAndCapacityDictList.append(idAndCapacityDict)
    
    return idAndCapacityDictList
