# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException

import common
import expandconfig
from cbb.business.operate.checkitems.check_required_hot_patch import \
    get_patch_ver_digital
from cbb.frame.cli import cliUtil


# noinspection PyUnresolvedReferences
LANG = common.getLang(py_java_env)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """ 检查硬盘域的成员盘数
        步骤1 以admin用户登录设备；
        步骤2 执行命令：show upgrade package，查询当前系统版本号；
        步骤3 执行命令：show disk_domain general disk_domain_id=?，查询硬盘域的
            成员盘数目；

        1、如果系统版本为Dorado V6 6.0.1.SPH2之后的版本，则检查通过。
        2、如果单个硬盘域中成员盘数目加上新扩盘数目大于400盘，检查不通过，否则检查通过。
    :param cli:
    :return:
    """
    check_process = MemberDiskNumCheck(cli, py_java_env, LANG, LOGGER)
    check_process.start()
    return check_process.get_result()


class MemberDiskNumCheck:
    # 单个硬盘域最大成员盘数
    LIMIT_DISK_NUM = 400

    def __init__(self, cli, py_java_env, lang, logger):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.env = py_java_env
        self.pdt_version = ""
        self.patch_version = ""
        self.full_version = ""
        self.result_flag = True
        self.all_ret_list = list()
        self.err_msg = ""
        self.is_new_dd_over = False
        self.is_exp_scene = self.is_exp_pre_check()

    def start(self):
        """ 开始执行检查

        :return:
        """
        try:
            # 扩容链路评估不涉及
            if self.is_exp_link():
                self.result_flag = cliUtil.RESULT_NOSUPPORT
                return

            # 初始化查询版本及补丁版本
            self.init_version()

            # 版本检查通过，返回
            if self.check_version():
                return

            # 获取硬盘域对应的成员盘数量信息
            domain_dict = self.get_domain_info()

            # 扩容评估场景下，获取扩容硬盘域扩容后的成员盘数量信息
            if self.is_exp_scene:
                domain_dict = self.get_exp_domain_dict(domain_dict)

            # 检查超过400的硬盘域
            over_limit_dd = self.check_over_limit_dd(domain_dict)
            # 如果有超过400盘硬盘域，或新建硬盘域超过400盘，不通过，生成错误消息
            if over_limit_dd or self.is_new_dd_over:
                self.generate_err_msg(over_limit_dd)
            return

        except (Exception, JException) as e:
            self.logger.logException(e)
            self.result_flag = cliUtil.RESULT_NOCHECK
            self.err_msg = common.getMsg(self.lang, "query.result.abnormal")
            return

    def init_version(self):
        """ 初始化查询系统版本及补丁版本

        :return:
        """
        flag, self.pdt_version, self.patch_version, version_ret, err_msg = \
            common.getProductVersionAndHotPatchVersion(self.cli, self.logger,
                                                       self.lang)
        self.all_ret_list.append(version_ret)
        if flag is not True:
            raise Exception("Failed to get version.")

        if self.patch_version and self.patch_version != "--":
            self.full_version = "{}.{}".format(self.pdt_version,
                                               self.patch_version)
        else:
            self.full_version = self.pdt_version

    def is_exp_pre_check(self):
        """是否是扩容评估场景

        :return:
        """
        scene_data = self.env.get("sceneData")
        return (scene_data
                and scene_data.get("mainScene") == "Expansion"
                and scene_data.get("toolScene") == "perInspector")

    def is_exp_link(self):
        """是否是扩容链路评估

        :return:
        """
        scene_data = self.env.get("sceneData")
        return (self.is_exp_scene
                and (scene_data.get("subScene") == "Expansion Links"
                     or self.env.get("expMode") == "EXTEND_LINK")
                )

    def get_domain_info(self):
        """ 获取硬盘域信息

        :return:
        """
        pool_cmd = "show storage_pool general"
        flag, ret, _ = cliUtil.excuteCmdInCliMode(self.cli, pool_cmd, True,
                                                  self.lang)
        self.all_ret_list.append(ret)
        if flag is not True:
            raise Exception("Failed to query storage pool info.")

        domain_dict = dict()
        pool_info_list = cliUtil.getHorizontalCliRet(ret)
        for pool_info in pool_info_list:
            dd_id = pool_info.get("Disk Domain ID")
            disk_num = self.get_disk_num_in_dd(dd_id)
            domain_dict[dd_id] = disk_num

        self.logger.logInfo("origin disk number: {}".format(domain_dict))
        return domain_dict

    def get_disk_num_in_dd(self, dd_id):
        """ 获取硬盘域中的硬盘数量

        :param dd_id: 硬盘域ID
        :return:
        """
        cmd = "show disk_domain general disk_domain_id={}".format(dd_id)
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True,
                                                        self.lang)
        self.all_ret_list.append(ret)
        if flag is not True:
            raise Exception("Failed to get disk num in domain[{}]".format(
                dd_id))

        info_list = cliUtil.getVerticalCliRet(ret)
        disk_num = int(info_list[0].get("Disk Number", 0))
        return disk_num

    def get_exp_domain_dict(self, domain_dict):
        """ 获取扩容场景硬盘域成员盘数量，包含未扩容的硬盘域

        :return:
        """
        exp_domain_dict = dict()
        exp_disk_list = expandconfig.ExpandConfig(self.env).getAllExpDiskList()
        for exp_disk in exp_disk_list:
            dd_id = exp_disk.getDiskDomainId()
            exp_disk_num = exp_disk.getDiskNum()
            if dd_id == "none":
                if exp_disk_num > self.LIMIT_DISK_NUM:
                    self.is_new_dd_over = True
            else:
                exp_domain_dict[dd_id] = exp_domain_dict.get(dd_id, 0) + \
                                         exp_disk_num
        for dd_id in domain_dict:
            if dd_id in exp_domain_dict:
                exp_domain_dict[dd_id] += domain_dict.get(dd_id)
            else:
                exp_domain_dict[dd_id] = domain_dict.get(dd_id)

        self.logger.logInfo("after expansion disk number: {}".format(
            exp_domain_dict))
        return exp_domain_dict

    def check_version(self):
        """ 检查版本及热补丁版本是否满足要求

        :return:
        """
        return (self.pdt_version == "6.0.1"
                and get_patch_ver_digital(self.patch_version) >=
                get_patch_ver_digital("SPH2"))

    def check_over_limit_dd(self, domain_dict):
        """ 检查超过400盘的硬盘域

        :return:
        """
        return [k for k in domain_dict
                if domain_dict.get(k, 0) > self.LIMIT_DISK_NUM]

    def generate_err_msg(self, dd_list):
        """ 生成错误消息

        :param dd_list:
        :return:
        """
        version_msg = common.getMsg(self.lang, "current.version",
                                    self.full_version)
        if self.is_exp_scene:
            dd_err_msg = "" if not self.is_new_dd_over else \
                common.getMsg(self.lang, "new.domain.disk.num.over")
            dd_err_msg_exp = "" if not dd_list else \
                common.getMsg(self.lang, "member.disk.num.over.after.exp",
                              ", ".join(dd_list))
            dd_err_msg += dd_err_msg_exp
        else:
            dd_err_msg = "" if not dd_list else \
                common.getMsg(self.lang, "member.disk.num.over.inspect",
                              ", ".join(dd_list))
        self.result_flag, self.err_msg = False, version_msg + dd_err_msg

    def get_result(self):
        """ 获取检查结果

        :return:
        """
        return self.result_flag, "\n".join(self.all_ret_list), self.err_msg
