# coding=UTF-8

# Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@version: Toolkit V200R006C00
@time: 2020/03/13
@file: check_item_software_pool_multi_dd_risk_check.py
@function:
@modify:
"""

import cliUtil
import common
from com.huawei.ism.tool.obase.exception import ToolException
import re


PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

FLAG_EXIST = "exist"
FLAG_NOT_EXIST = "not exist"
# 检查通过的软件版本
CHECK_PASS_SOFT_VERSION = "V300R001"
# 检查通过的软件版本和对应补丁版本
CHECK_PASS_SOFT_AND_PATCH_VERSION = {
    "V300R002C00SPC100": [105, 170],  # 列表中为热补丁的范围，左闭右开区间。
    "V300R002C10SPC100": [110, 170],
    "V300R002C20": [2, 70]
}
# 根据disk num计算raid的大小的判断范围
RAID_DISK_NUM_STEP = {
    (6, 13): -1,
    (13, 26): -2,
    (26, 28): -3,
    (28, 29): -3,
}
# 超出范围的raid
MAX_RAID = 25

# 额外的巡检判断通过的特殊版本，不在上边的范围，但是也要通过
CHECK_PASS_SPECIAL_VERSION = ("V300R002C10SPC100", 177)

PARE_PATCH_NUM_REG = r"(\d+)$"  # 提取出热补丁的版本号数字部分
# Cli指令
CMD_STORAGE_POOL_INFO = "show storage_pool general"
CMD_DISK_DOMAIN_INFO = "show disk_domain general"
CMD_DISK_DOMAIN_DETAIL = "show disk_domain general disk_domain_id={domain_id}"
# 标识检查结果
CHECK_RESULT_CONTINUE = "continue"

# 关系计算常量1296000*4
CONST_RELATION_SHIP = 5184000


def execute(cli):
    """
    引擎内多硬盘域风险检查
    """
    all_cli_ret = ""
    try:
        # 1.判断版本信息
        flag, cli_ret, err_msg = check_soft_ver_and_patch_ver(cli)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        if flag != CHECK_RESULT_CONTINUE:  # 通过或者异常
            return flag, all_cli_ret, err_msg

        # 2.判断是否存在硬盘域
        flag, cli_ret, err_msg, domain_id_list = check_exist_disk_domain(cli)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        if flag != CHECK_RESULT_CONTINUE:  # 通过或者异常
            return flag, all_cli_ret, err_msg
        LOGGER.logInfo("domain id list: {0}".format(domain_id_list))

        # 3.判断是否有同一个控制器存在多个硬盘域
        flag, cli_ret, err_msg, same_controller_info = \
            exist_same_controller_domain(cli, domain_id_list)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        if flag != FLAG_NOT_EXIST and flag != FLAG_EXIST:  # 执行失败
            return flag, all_cli_ret, err_msg
        elif flag == FLAG_NOT_EXIST:
            return True, all_cli_ret, err_msg
        LOGGER.logInfo("same controller: {0}".format(same_controller_info))

        # 4. 以下逻辑为存在相同控制器的硬盘域的情况下，判断是否存在存储池信息
        flag, cli_ret, err_msg, storage_pool_info = exist_storage_pool(cli)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        if flag != FLAG_NOT_EXIST and flag != FLAG_EXIST:  # 执行失败
            return flag, all_cli_ret, err_msg
        elif flag == FLAG_NOT_EXIST:  # 不存在存储池信息, 建议优化
            err_msg = pare_warning_msg(same_controller_info)
            return cliUtil.RESULT_WARNING, all_cli_ret, err_msg
        LOGGER.logInfo("storage pool: {0}".format(storage_pool_info))

        # 5. 相同控制器的硬盘域中，最多只有一个硬盘域存在存储池
        flag, exist_pool_same_controller = check_only_one_exist_pool_per_same(
            same_controller_info, storage_pool_info)
        if flag:  # 所有相同控制器组中的硬盘域只有最多一个存在存储池，建议优化
            err_msg = pare_warning_msg(same_controller_info)
            return cliUtil.RESULT_WARNING, all_cli_ret, err_msg

        LOGGER.logInfo(
            "same controller exist pool: {0}".format(
                exist_pool_same_controller))
        same_controller_group = {}

        # 过滤出大于等于2的相同控制器且有存储池的
        for controller, domain_list in exist_pool_same_controller.iteritems():
            if len(domain_list) >= 2:
                same_controller_group[controller] = domain_list

        # 6. 以下为存在相同控制器的硬盘域，且他们有存在存储池的情况。
        flag, err_msg = get_same_controller_check_result(same_controller_group)
        if flag is True:  # 表示不满足表格，但是需要建议优化
            err_msg = pare_warning_msg(same_controller_info)
            return cliUtil.RESULT_WARNING, all_cli_ret, err_msg

        # 不通过
        return flag, all_cli_ret, err_msg
    except (Exception, ToolException) as e:
        LOGGER.logException(e)
        return (cliUtil.RESULT_NOCHECK, all_cli_ret,
                common.getMsg(LANG, "query.result.abnormal"))


def check_soft_ver_and_patch_ver(cli):
    """
    检查标准1中软件版本和热补丁的判断，满足条件则通过，否则继续检查
    :return , False, "NOCHECK"
    """
    flag, soft_ver, patch_ver, cli_ret, err_msg = \
        common.get_soft_and_patch_version(cli, LOGGER, LANG)
    # 命令没有执行成功
    if flag is not True:
        return flag, cli_ret, err_msg

    LOGGER.logInfo(
        "soft_ver:{0}, patch_ver:{1}".format(soft_ver, patch_ver))

    if soft_ver.startswith(CHECK_PASS_SOFT_VERSION):
        return True, cli_ret, err_msg

    matched_patch_ver = CHECK_PASS_SOFT_AND_PATCH_VERSION.get(soft_ver, [])
    patch_ver_num_result = re.findall(r"SPH(\d+)$", patch_ver)
    # 软件版本在匹配字典中，并且有补丁版本
    if matched_patch_ver and patch_ver_num_result:
        patch_ver_num = int(patch_ver_num_result[0])
        # 补丁版本满足范围条件
        if matched_patch_ver[0] <= patch_ver_num < matched_patch_ver[1]:
            return True, cli_ret, err_msg
        elif (soft_ver, patch_ver_num) == CHECK_PASS_SPECIAL_VERSION:
            return True, cli_ret, err_msg

    return CHECK_RESULT_CONTINUE, cli_ret, err_msg


def check_exist_disk_domain(cli):
    """
    判断是否存在硬盘域，不存在则直接通过，否则继续
    :param cli:
    :return:
    """
    disk_domain_id_list = []
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
        cli, CMD_DISK_DOMAIN_INFO, True, LANG)
    if flag is not True:
        return flag, cli_ret, err_msg, []

    # 没有硬盘域信息
    if cliUtil.no_record(cli_ret):
        return True, cli_ret, err_msg, []

    cli_ret_line = cliUtil.getHorizontalCliRet(cli_ret)
    for line in cli_ret_line:
        disk_domain_id = line.get("ID")
        disk_domain_id_list.append(disk_domain_id)
    return CHECK_RESULT_CONTINUE, cli_ret, err_msg, disk_domain_id_list


def exist_same_controller_domain(cli, domain_id_list):
    """
    判断是否存在同一个控制器有多个存储域
    :return controller_domain_info = {
        "controller": [
            {"domain_id": "硬盘域id"， "disk_number": "硬盘域的硬盘数"}
        ]
    }
    """
    controller_domain_info = {}
    all_cli_ret = ""
    for domain_id in domain_id_list:
        cmd = CMD_DISK_DOMAIN_DETAIL.format(domain_id=domain_id)
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            cli, cmd, True, LANG)
        all_cli_ret = common.joinLines(all_cli_ret, cli_ret)
        # 命令没有执行成功
        if flag is not True:
            return flag, all_cli_ret, err_msg, {}

        disk_domain_detail_list = cliUtil.getVerticalCliRet(cli_ret)
        if not disk_domain_detail_list:  # 命令执行成功，但没有数据，理论不应存在此情况
            msg = common.getMsg(LANG, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK, all_cli_ret, msg, {}

        disk_domain_detail = disk_domain_detail_list[0]
        dis_number = disk_domain_detail.get("Disk Number", 0)
        controller = disk_domain_detail.get("Controller", "")
        domain_info = {
            "domain_id": domain_id,
            "disk_number": dis_number
        }
        # 将相同控制器的硬盘域信息归纳在一起
        if controller in controller_domain_info:
            controller_domain_info[controller].append(domain_info)
        else:
            controller_domain_info[controller] = [domain_info]

    # 统计出相同控制器的硬盘域列表大于等于2的。
    same_controller_group = []
    for controller, info in controller_domain_info.iteritems():
        if len(info) >= 2:
            same_controller_group.append(info)

    if not same_controller_group:  # 即所有硬盘域的控制器不存在相同的情况
        return FLAG_NOT_EXIST, all_cli_ret, err_msg, {}

    return FLAG_EXIST, all_cli_ret, err_msg, controller_domain_info


def exist_storage_pool(cli):
    """
    判断是否存在存储池，如果有存储池，则返回以存储池的disk domain id为key的存储池信息
    :param cli:
    :return: {"domain_id": {"pool_id": “存储池id”, "pool_cap": "存储池总容量"}}
    """
    storage_pool_info = {}
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
        cli, CMD_STORAGE_POOL_INFO, True, LANG)
    if flag is not True:
        return flag, cli_ret, err_msg, {}

    # 没有存储池信息
    if cliUtil.no_record(cli_ret):
        LOGGER.logInfo("no storage pool exist")
        return FLAG_NOT_EXIST, cli_ret, err_msg, {}

    cli_ret_line = cliUtil.getHorizontalCliRet(cli_ret)
    for pool_info in cli_ret_line:
        domain_id = pool_info.get("Disk Domain ID", None)
        if domain_id:
            origin_cap = pool_info.get("Total Capacity", "0TB")
            pool = {
                "pool_cap": cliUtil.transfer_capacity(origin_cap, "MB"),
                "pool_id": pool_info.get("ID"),
            }
            storage_pool_info[domain_id] = pool

    return FLAG_EXIST, cli_ret, err_msg, storage_pool_info


def check_only_one_exist_pool_per_same(controller_info, storage_pool_info):
    """
    检查每一组相同控制器的硬盘域中，是否每一组只有最多一个硬盘域存在存储池
    """
    result = []
    # 存在存储池的并且有相同控制器的数据
    exist_pool_same_controller = {}
    for controller, domain_list in controller_info.iteritems():
        count = 0
        for domain in domain_list:
            domain_id = domain['domain_id']
            # 存储池对应的硬盘域id在相同控制器中
            if domain_id in storage_pool_info:
                count += 1
                domain.update(**storage_pool_info[domain_id])
                if controller in exist_pool_same_controller:
                    # 将存储池信息也添加到domain信息中
                    exist_pool_same_controller[controller].append(domain)
                else:
                    exist_pool_same_controller[controller] = [domain]
        result.append(count)
    if set(result) == set([0, 1]):
        return True, {}
    return False, exist_pool_same_controller


def get_same_controller_check_result(same_controller_group):
    """
    :param same_controller_group: 相同控制器的硬盘域信息
    :return: True、False.True：建议优化，False：不通过
    """
    error_msg = []
    # 对相同控制器的硬盘域作计算，判断是否优化
    for controller, domain_list in same_controller_group.iteritems():
        # 控制器个数
        controller_count = len(controller.split(","))
        # 引擎个数等于控制器个数除以2
        engine_count = controller_count / 2
        for index in xrange(len(domain_list)):
            # 取出剩下的硬盘域信息进行计算
            other_domain_list = domain_list[:index] + domain_list[index+1:]
            min_disk_number = min(
                [int(domain['disk_number']) for domain in other_domain_list])
            # 根据控制器个数，算出单个引擎的最小硬盘数
            single_engine_min_disk_number = min_disk_number / engine_count
            # 先默认raid等于最大值
            len_raid = MAX_RAID
            # 若满足范围条件再更新len_raid
            for step, val in RAID_DISK_NUM_STEP.iteritems():
                if step[0] <= single_engine_min_disk_number < step[1]:
                    len_raid = single_engine_min_disk_number + val
                    break

            sum_capacity = sum(
                [domain['pool_cap'] for domain in other_domain_list])
            sum_capacity_one_engine = sum_capacity / engine_count
            LOGGER.logInfo(
                "sum_capacity:{0},len_raid:{1},"
                "min_disk_number:{2},controller:{3}".format(
                    sum_capacity, len_raid, min_disk_number, controller))
            # 一旦满足此条件，就不通过
            if sum_capacity_one_engine >= CONST_RELATION_SHIP*len_raid:
                domain_id = ",".join(
                    [domain['domain_id'] for domain in domain_list])
                msg = common.getMsg(
                    LANG, "if_software_pool_multi_dd_risk.notpass", (
                        domain_id, controller))
                if msg not in error_msg:
                    error_msg.append(msg)
    error_msg = "".join(error_msg)
    if error_msg:  # 不通过
        return False, error_msg
    return True, error_msg


def pare_warning_msg(same_controller_info):
    """
    组装waning消息
    :param same_controller_info:
    :return:
    """
    error_msg = []
    for controller, domain_list in same_controller_info.iteritems():
        domain_id = ",".join([domain['domain_id'] for domain in domain_list])
        msg = common.getMsg(
            LANG, "if_software_pool_multi_dd_risk.warning", (
                domain_id, controller))
        error_msg.append(msg)

    error_msg = "".join(error_msg)
    return error_msg
