# -*- coding: UTF-8 -*-
from __future__ import division
import re
import decimal
import cliUtil
import common

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

def execute(cli):
    '''
        巡检工具增加配置smartTier特性后存储池空间预留的检查提示 ； 
        步骤1 以admin用户登录设备；
        步骤2 执行show upgrade package命令，获取系统版本；
        步骤3 执行show schedule命令，获取类型为自动迁移（Type为Automatic Relocation）的Pool ID；
        步骤4 执行show storage_pool general pool_id=?命令，
                        计算配置有SmartTier的存储池空闲容量（Free Capacity） / 总容量（Total Capacity），
                        若小于等于10%，则该存储池检查建议优化，若大于10%，则该存储池检查通过。
    '''
    
    allCliRet = ""
    errMsg = ""
    freePercent = 0.1
    lowCapPoolIDList = []
    errList = []
    
    try:
        #获取类型为自动迁移的Pool ID列表
        execRet = getAutoRelocationIDList(cli)
        allCliRet += "\n"+execRet[1]
        poolIDList = execRet[-1]
        
        #获取Pool ID列表为空
        if execRet[0] != True:
            LOGGER.logNoPass("get pool ID failed,error message:%s" % execRet[2])
            errMsg = execRet[2]
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)  # 修改备注：命令执行失败是未检查
        
        #检查所有存储池的空闲容量
        for poolID in poolIDList:
            checkRet = checkFreeCapacity(cli, poolID, freePercent)
            allCliRet += "\n" + checkRet[1]
            if checkRet[0]==False :
                errMsg += checkRet[-1]
                errList.append(poolID)
            if checkRet[0]==cliUtil.RESULT_WARNING:
                lowCapPoolIDList.append(poolID)
            
        
        #有存储池检查失败
        if errList:
            errIDStr = ",".join(errList)
            errMsg += common.getMsg(LANG,"error.during.checking.free.capacity",errIDStr)
            return (False, allCliRet, errMsg)
        
        #有空闲容量低于10%的存储池
        if lowCapPoolIDList:
            lowCapPoolIDStr = ",".join(lowCapPoolIDList)
            errMsg += common.getMsg(LANG,"pool.low.capacity",lowCapPoolIDStr)
            return (cliUtil.RESULT_WARNING, allCliRet, errMsg)
        
        return (True, allCliRet, errMsg)
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))

def getAutoRelocationIDList(cli, lang=LANG):
    """
    @summary: 获取类型为自动迁移（Type为Automatic Relocation）的Pool ID列表
    @param cmd: 执行的命令
    @return: 
        isSucc：True/False，方法是否正常结束
        allCliRet：所有CLI回显
        errMsg：方法异常结束时的原因
        poolIDList: 类型为自动迁移的Pool ID列表
    """
    
    isSucc = True
    cliRet = ""
    errMsg = ""
    poolIDList = []
    
    try:
        cmd = "show schedule"
        execRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        cliRet = execRet[1]
        #命令执行失败
        if execRet[0] != True:
            isSucc = execRet[0]
            errMsg = execRet[2]+"\n"+common.getMsg(LANG,"pool.id.null")
            return (isSucc,cliRet,errMsg,poolIDList)
        
        
        if cliUtil.queryResultWithNoRecord(cliRet):
            return (True,cliRet,errMsg,poolIDList)
        
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
        for line in cliRetLinesList:
            type = line.get("Type")
            poolID = line.get("Pool ID")
            if type=="Automatic Relocation":
                poolIDList.append(poolID)
                
        return (isSucc,cliRet,errMsg,poolIDList)
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "query.result.abnormal"),[])  # 修改备注：捕捉到异常是未检查

def checkFreeCapacity(cli, poolID, freePercent, lang=LANG):
    """
    @summary: 检查存储池空闲容量
    @param poolID: 存储池ID
    @return: 
        isSucc：True/False，方法是否正常结束
        allCliRet：所有CLI回显
        errMsg：方法异常结束时的原因
        isPass: 空闲容量占总容量百分比是否超过10%（超过为True，否则False）
    """
    isSucc = True
    cliRet = ""
    errMsg = ""
    
    try:
        cmd = "show storage_pool general pool_id=%s" % str(poolID)
        execRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        cliRet = execRet[1]
        if execRet[0] != True:
            LOGGER.logNoPass("cliRet is abnormal,error message:%s" % execRet[-1])
            errMsg = execRet[-1]
            return (execRet[0],cliRet,errMsg)
        
        cliList = cliUtil.getVerticalCliRet(cliRet)
        if not cliList:
            LOGGER.logNoPass("get cliRet failed")
            isSucc = False
            return (isSucc,cliRet,execRet[-1])
        
        statusDict = cliList[0]
        freeCap = statusDict.get("Free Capacity")
        totalCap = statusDict.get("Total Capacity")
        freeRet = common.changUnit2GBDecimal(freeCap)
        totalRet = common.changUnit2GBDecimal(totalCap)
        
        #获取空闲容量或总容量失败
        if freeRet[0] != True:
            errMsg += common.getMsg(LANG, "get.free.capacity.failed")
            return (False, cliRet, errMsg)
        if totalRet[0] != True or totalRet[1] == decimal.Decimal("0.0"):
            errMsg += common.getMsg(LANG, "get.total.capacity.failed")
            return (False, cliRet, errMsg)
        
        freeNum = freeRet[1]
        totalNum = totalRet[1]
        if (freeNum/totalNum) <= decimal.Decimal(str(freePercent)):
            isSucc = cliUtil.RESULT_WARNING
        
        return (isSucc, cliRet, errMsg)
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (False, cliRet, common.getMsg(LANG, "query.result.abnormal"))