# -*- coding: UTF-8 -*-

import re

from cbb.frame.checkitem.context_adapter import InspectContext
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN
from cbb.frame.dsl import fault_mode as ft

from common import AsynProgress
import common_utils
import common
from risk_version_config import STORAGE_POOL_PARAM_RISK_VERSION

PY_JAVA_ENV = py_java_env
JAVA_LOGGER = PY_LOGGER
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(JAVA_LOGGER, __file__)


def execute(cli):
    # 进度条刷新
    my_thread = AsynProgress(PY_JAVA_ENV, LOGGER)
    # 进度开始
    my_thread.start_thread()
    context = InspectContext(PY_JAVA_ENV, cli)
    inspect_context = context.get_context()
    inspect_context["logger"] = JAVA_LOGGER
    inspect_context["lang"] = LANG
    check_obj = CheckItem(inspect_context, cli)
    status, err_msg = check_obj.check()
    all_ret = check_obj.get_origin_info()
    LOGGER.logInfo("status={},err_msg={}".format(status, err_msg))
    my_thread.setStopFlag(True)
    return context.get_result(status, err_msg, all_ret)


class CheckItem(BaseCheckItem):
    def __init__(self, context, cli):
        super(CheckItem, self).__init__(context)
        self.dev_version = ""
        self.patch_version = ""
        self.context[NEED_NEW_CTRL_CONN] = True
        self.cli = cli
        self.env = PY_JAVA_ENV
        self.local_logger = LOGGER
        self.storage_pool_id_list = []
        self.pool_info_list = []

    @common_utils.check_conn_wrap
    def execute(self):
        self.init_version_and_patch()
        if not self.is_risk_version():
            return CheckStatus.PASS, ""

        self.query_all_storage_pool()

        if not self.pool_info_list:
            return CheckStatus.PASS, ""

        if not self.storage_pool_id_list:
            return CheckStatus.NOTPASS, self.deal_suggestion_msg()

        if not common_utils.is_super_administrator(
                self.cli, self.env.get("devInfoMap").get("userName"), LANG
        ):
            return CheckStatus.NOCHECK, common.getMsg(
                LANG, "user.level.not.super_admin")

        return self.check_all_controller()

    def check_all_controller(self):
        """
        检查全部控制器
        :return:
        """
        all_ctrl_result = self.dsl(
            "exec_on_all {}",
            self.check_cache_inst_id,
            is_continue=False,
            need_node_id=True,
        )
        status, err_msg = CheckStatus.merge_all_ctrl_result(
            self.lang, all_ctrl_result
        )
        if status == CheckStatus.NOTPASS:
            err_msg = self.deal_suggestion_msg() + err_msg

        return status, err_msg

    def init_version_and_patch(self):
        """
        初始化版本、补丁信息
        :return:
        """
        (
            flag,
            self.dev_version,
            self.patch_version,
            ret,
        ) = replace_adapter.get_system_version_with_ret(self.context)
        self.add_origin_info(ret)

    def query_all_storage_pool(self):
        """
        查询全部存储池
        :return:
        """
        self.pool_info_list = self.dsl(
            "exec_cli 'show storage_pool general |filterColumn include "
            "columnList=ID,Disk\sDomain\sID' | horizontal_parser",
        )

        self.storage_pool_id_list = [
            pool.get("ID")
            for pool in self.pool_info_list
            if pool.get("Disk Domain ID") == "0"
        ]

    def check_cache_inst_id(self):
        """
        检查 cacheInstId 的值是否都为 0
        :return:
        """
        error_pool_msg = []
        for pool_id in self.storage_pool_id_list:
            ret = self.dsl(
                "exec_diagnose 'storagepool showstoragepoolinfo {}'".format(
                    pool_id),
                return_if={ft.FindException: ''}
            )
            self.local_logger.logInfo("got the ret is:{}".format(ret))
            res = re.compile("cacheInstId\s*:\s*\((\w+)\)").findall(ret)
            if not res:
                return (
                    CheckStatus.NOCHECK,
                    common.getMsg(LANG, "query.result.abnormal"),
                )

            if res[0] == "0":
                continue

            error_pool_msg.append(
                "Pool {} cacheInstId is:{}".format(pool_id, res[0]))

        if error_pool_msg:
            return CheckStatus.NOTPASS, ", ".join(error_pool_msg)

        return CheckStatus.PASS, ''

    def deal_suggestion_msg(self):
        """
        根据版本处理修复建议
        :return:
        """
        if self.dev_version == "6.0.RC1":
            return common_utils.get_err_msg(
                LANG,
                "storagepool.param.check.not.pass.upgrade.version",
                (self.dev_version, "6.0.1.SPH8"),
            )

        target_patch = STORAGE_POOL_PARAM_RISK_VERSION.get(self.dev_version)
        if not target_patch:
            return common_utils.get_err_msg(
                LANG,
                "storagepool.param.check.not.pass.upgrade.version",
                (self.dev_version, "6.1.RC3"),
            )
        return common_utils.get_err_msg(
            LANG,
            "storagepool.param.check.not.pass.install.patch",
            (self.dev_version, target_patch),
        )

    def is_risk_version(self):
        """
        Dorado V6 6.0.0.SPH11之前版本,Dorado V6 6.0.1.SPH8之前版本,
        Dorado V6 6.1.RC1,Dorado V6 6.1.RC2
        :return: True 风险版本, False 非风险。
        """
        if self.dev_version not in STORAGE_POOL_PARAM_RISK_VERSION:
            return False

        target_patch = STORAGE_POOL_PARAM_RISK_VERSION.get(self.dev_version)
        if not target_patch:
            return True

        if common_utils.get_patch_value(self.patch_version) >= \
                common_utils.get_patch_value(target_patch):
            return False
        return True
