# -*- coding: UTF-8 -*-
import cliUtil
import common
import config
from cbb.common.conf.productConfig import EXPAND_TIER_CHECK_PDT_MODEL
from cbb.common.conf.productConfig import EXPAND_TIER_CHECK_PDT_MODEL_HIGH

HDD_MIN_4_MID = 4
HDD_MIN_4_HIGH = 8
SSD_MIN_HAVE_HDD_4_MID = 2
SSD_MIN_NO_HDD_4_MID = 4
SSD_MIN_HAVE_HDD_4_HIGH = 6
SSD_MIN_NO_HDD_4_HIGH = 8


class ExpTask:
    
    def __init__(self, py_java_env, cli, expDiskList, logger):
        '''
        @param expDiskList: 同一硬盘域，同一引擎的新扩硬盘信息
        '''
        self.allCliRet = ""
        self.errMsg = ""
        self.result = True
        
        self.logger = logger
        self.lang = common.getLang(py_java_env)
        self.py_java_env = py_java_env
        self.productModel = str(common.getProductModeFromContext(py_java_env))
        self.initCli(cli)
        self.expDiskList = expDiskList
        self.domainId = self.getDomainId()
        self.diskList = self.getDiskOnDomain()
        self.expSSDList, self.expHDDList = self.getExpDiskInfo()

    def initCli(self, cli):
        self.logger.logInfo("start get cli connect.")
        if common.is18000(self.py_java_env, cli):
            ret = common.getCli4Ctrl(cli, self.lang, self.logger, self.py_java_env)
            self.logger.logInfo("get cli connect result:%s"%str(ret))
            self.cli = ret[2]
        else:
            self.cli = cli
        return

    def releaseCli(self):
        if common.is18000(self.py_java_env, self.cli):
            self.py_java_env.get("sshManager").releaseConnection(self.cli)
        return

    def execute(self):
        try:
            if self.result != True:
                return

            #获取存储池信息
            self.storagePools = self.getStoragePools()
            if self.storagePools == None:
                return

            #获取存储池tier信息
            self.tierInfo = self.getTierInfo(self.storagePools)
            if self.tierInfo == None:
                return

            #获取存在HDD，SSD的引擎
            self.hddEngList, self.ssdEngList = self.getDiskEngs()

            self.checkExpHDD()
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.logger)
            self.checkExpSSD()
            return
        finally:
            self.releaseCli()
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.logger)


    def checkExpHDD(self):
        '''
        @summary: 检查新扩的HDD盘是否满足要求
        '''
        #F系列不检查
        if self.isFDev():
            self.logger.logInfo("The current device is F device:%s" %self.productModel)
        
        newTier = self.getNewTier4HDD()
        self.logger.logInfo("The new tier of the domain(%s) is:%s."%(self.domainId, newTier))
        if newTier:
            hddDiskNumMap = self.getExpHDDNum(newTier)
            
            for (diskType, hddDiskNum) in hddDiskNumMap.items():
                if self.productModel in EXPAND_TIER_CHECK_PDT_MODEL \
                        and hddDiskNum < HDD_MIN_4_MID:
                    self.result = False
                    #在硬盘域%s上，新扩容的HDD硬盘数量%s不满足最小起扩盘数%s。
                    self.errMsg += common.getMsg(self.lang, "exp.tire.hdd.disk.too.less",(self.domainId, diskType, hddDiskNum, HDD_MIN_4_MID))
                
                if self.productModel in EXPAND_TIER_CHECK_PDT_MODEL_HIGH \
                        and hddDiskNum < HDD_MIN_4_HIGH:
                    self.result = False
                    #在硬盘域%s上，新扩容的HDD硬盘数量%s不满足最小起扩盘数%s。
                    self.errMsg += common.getMsg(self.lang, "exp.tire.hdd.disk.too.less",(self.domainId, diskType, hddDiskNum, HDD_MIN_4_HIGH))
        for hddDisk in self.expHDDList:
            self.checkOneExpDisk(hddDisk)
            
        return
    
    def checkExpSSD(self):
        '''
        @summary: 检查新扩的SSD盘是否满足要求
        '''
        
        for ssdDisk in self.expSSDList:
            expEng = ssdDisk[0]
            diskType = ssdDisk[1]
            diskNum = ssdDisk[2]
            newTierNum = self.getNewTierSSDNum(ssdDisk)
            raidDiskNum = 0
            if self.existDiskType(expEng, diskType):
                raidDiskNum = self.getMaxRaidDiskNum(diskType)
                self.logger.logInfo("Get raid disk number:%s" %raidDiskNum)
                if raidDiskNum == None:
                    continue
            
            minNum = newTierNum if raidDiskNum < newTierNum else raidDiskNum
            if not minNum:
                self.logger.logInfo("expand existed disk type:%s" %diskType)
                continue
            
            if diskNum < minNum:
                self.result = False
                dispEng = common.getDisplayEngId(expEng, self.productModel)
                #在硬盘域%s中，引擎%s上新扩容的硬盘（类型：%）数量（数量：%s）不满足最小起扩盘数（数量：%s）。
                self.errMsg += common.getMsg(self.lang, "exp.tire.exp.disk.less",(self.domainId, dispEng, diskType, diskNum, minNum))
                continue

    def getExpDiskInfo(self):
        '''
        @summary: 格式化新扩容硬盘信息[(eng,diskType,diskNum)]
        '''
        expEngDict = {}
        for expDisk in self.expDiskList:
            logiceEng = expDisk.getDiskLogicEngId()
            expDiskType = expDisk.getDiskType()
            expDiskNum = expDisk.getDiskNum()
            diskInfo = expEngDict.get(logiceEng,{})
            diskInfo[expDiskType] = diskInfo.get(expDiskType, 0) + expDiskNum
            expEngDict[logiceEng] = diskInfo
            
        expSSDList = []
        expHDDList = []
        expEngs = sorted(expEngDict.keys())
        for expEng in expEngs:
            expdiskDict = expEngDict.get(expEng)
            diskTypes = expdiskDict.keys()
            for diskType in diskTypes:
                if self.isHDDDisk(diskType):
                    expHDDList.append((expEng, diskType, expdiskDict.get(diskType)))
                else:
                    expSSDList.append((expEng, diskType, expdiskDict.get(diskType)))
                
        return expSSDList, expHDDList

    def getDomainId(self):
        if self.expDiskList:
            return self.expDiskList[0].getDiskDomainId()
        return None
    
    def getDiskOnDomain(self):
        '''
        @summary: 获取当前硬盘域
        '''
        if not self.domainId:
            return []
        
        cmd = "show disk in_domain disk_domain_id=%s|filterColumn include columnList=ID,Type" %self.domainId
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.allCliRet += cliRet
        if flag != True:
            self.setResult(cliUtil.RESULT_NOCHECK)
            self.errMsg = errMsg
            
        return cliUtil.getHorizontalCliRet(cliRet)
    
    def isHDDDisk(self,diskType):
        '''
        @summary: 为HDD返回True，否则返回False，非HDD即为SSD
        '''
        if "SSD" not in str(diskType.upper()):
            return True
        
        return False
    
    def getNewTier4HDD(self):
        '''
        @summary: 从扩容配置中获取新增层级的HDD
        '''
        
        oldTier = set()
        for disk in self.diskList:
            diskType = disk.get("Type")
            tier = config.DISK_TYPE_LEVEL_KEY.get(diskType)
            oldTier.add(tier)
            
        newTier = set()
        for expDisk in self.expHDDList:
            expDiskType = expDisk[1]
            tier = config.DISK_TYPE_LEVEL_KEY.get(expDiskType)
            if tier not in oldTier:
                newTier.add(expDiskType)
                
        return newTier
    
    def getExpHDDNum(self, newTier):
        '''
        @summary: 获取新增层级HDD盘的数量（即非SSD盘）
        '''
        diskNumMap = {}
        for expDisk in self.expDiskList:
            expDiskType = expDisk.getDiskType()
            if expDiskType in newTier and self.isHDDDisk(expDiskType):
                diskNumMap[expDiskType] = expDisk.getDiskNum()
        return diskNumMap
    
    def isFDev(self):
        # 如果产品型号为F系列，检查结果为不涉及
        return self.productModel not in EXPAND_TIER_CHECK_PDT_MODEL \
               and self.productModel not in EXPAND_TIER_CHECK_PDT_MODEL_HIGH
    
    def getDiskEngs(self):
        '''
        @summary: 获取存在HDD盘,SSD盘的逻辑引擎，HDD包含新扩HDD，SSD不包含新扩SSD
        '''
        hddEngList = set()
        ssdEngList = set()
        for disk in self.diskList:
            diskType = disk.get("Type")
            logicEng = common.getEngineIdByDiskId(disk.get("ID"),self.productModel)
            if self.isHDDDisk(diskType):
                hddEngList.add(logicEng)
            else:
                ssdEngList.add(logicEng)
        
        for expDisk in self.expHDDList:
            hddEngList.add(expDisk[0])
            
        return hddEngList,ssdEngList
    
    def existDiskType(self, expLogicEng, expDiskType):
        '''
        @summary: 1、原硬盘域存在新扩的硬盘类型，2、新扩目标逻辑引擎不存在新扩的硬盘类型
        @return: 满足1,2返回 True，
        '''
        existDiskOnDomain = False
        existDiskOnExpLogicEng = False
        
        for disk in self.diskList:
            diskId = disk.get("ID")
            diskType = disk.get("Type")
            logicEng = common.getEngineIdByDiskId(diskId, self.productModel)
            if diskType == expDiskType:
                existDiskOnDomain = True
            
                if str(logicEng) == str(expLogicEng):
                    existDiskOnExpLogicEng = True
            
            if existDiskOnDomain and existDiskOnExpLogicEng:
                return False
        
        if existDiskOnDomain and not existDiskOnExpLogicEng:
            return True
        
        return False
        
    def getStoragePools(self):
        '''
        @summary: 获取storage pool信息
        @return: None:获取失败；其他获取成功
        '''
        
        cmd = "show storage_pool general |filterRow column=Disk\sDomain\sID predict=equal_to value=%s" % self.domainId
        checkRet = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        cliRet = checkRet[1]
        self.allCliRet += cliRet
        if not checkRet[0]:
            #获取硬盘域（ID：%s）的存储池信息异常。
            self.setResult(cliUtil.RESULT_NOCHECK)
            self.errMsg += common.getMsg(self.lang, "exp.tire.storage.pool.fail",self.domainId)
            return None
        
        if cliUtil.queryResultWithNoRecord(cliRet):
            return []
        
        poolDictList = cliUtil.getHorizontalCliRet(cliRet)
        if len(poolDictList) == 0:
            self.setResult(cliUtil.RESULT_NOCHECK)
            self.errMsg += common.getMsg(self.lang, "exp.tire.storage.pool.fail",self.domainId)
            return None
        
        return poolDictList
    
    def getTierInfo(self, storagePools):
        
        tierInfo = []
        for pool in storagePools:
            
            cmd = "show storage_pool tier pool_id=%s|filterColumn include columnList=Name,RAID\sDisk\sNumber" %pool.get("ID")
            checkRet = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
            cliRet = checkRet[1]
            self.allCliRet += cliRet
            if not checkRet[0]:
                #获取硬盘域（ID：%s）的存储池层级信息异常。
                self.setResult(cliUtil.RESULT_NOCHECK)
                self.errMsg += common.getMsg(self.lang, "exp.tire.storage.pool.tier.fail",self.domainId)
                return None
            
            if cliUtil.queryResultWithNoRecord(cliRet):
                return []
            
            poolDictList = cliUtil.getVerticalCliRet(cliRet)
            if len(poolDictList) == 0:
                self.setResult(cliUtil.RESULT_NOCHECK)
                self.errMsg += common.getMsg(self.lang, "exp.tire.storage.pool.tier.fail",self.domainId)
                return None
            
            tierInfo.extend(poolDictList)
        
        return tierInfo
    

    def getHotDiskNum(self, raidNum):
        '''
        @summary: 获取热备盘数
        '''
        if raidNum in range(1, 13):
            return 1
        
        if raidNum in range(13, 25):
            return 2
        
        if raidNum in range(25, 29):
            return 3
        
        return 0
        
    def getMetaRaidDiskNum(self,diskType):
        '''
        @summary: 获取元数据的raid盘数，V3R3C10SPC100及之后版本适用
        @return: 返回None则获取异常
        '''
        
        checkMetaSpace = False
        for pool in self.storagePools:
            
            usageType = pool.get("Usage Type","LUN").upper()
            if usageType != "LUN":
                checkMetaSpace = True
                break
            
            poolId = pool.get("ID")
            cmd = "show lun dedup_compress pool_id=%s"%poolId
            result, cliRet, __=cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
            self.allCliRet += cliRet
            if not cliUtil.hasCliExecPrivilege(cliRet):
                return 0
            
            if cliUtil.queryResultWithNoRecord(cliRet):
                continue
            
            if "does not exist" in cliRet:
                continue
            
            if result != True:
                self.setResult(cliUtil.RESULT_NOCHECK)
                #获取硬盘域（%s）最大RAID成员盘数异常
                self.errMsg += common.getMsg(self.lang, "exp.tire.storage.pool.meta.tier.fail",self.domainId)
                return None
            
            lunList = cliUtil.getHorizontalCliRet(cliRet)
            if lunList:
                checkMetaSpace = True
                break
        
        if not checkMetaSpace:
            return 0
        
        cmd = "show disk_domain meta_space_utilization disk_domain_id=%s" %self.domainId
        result, cliRet, __= cliUtil.excuteCmdInDeveloper(self.cli, cmd, True, self.lang)
        self.allCliRet += cliRet

        if not cliUtil.hasCliExecPrivilege(cliRet):
            return 0
        
        if result != True:
            self.setResult(cliUtil.RESULT_NOCHECK)
            #获取硬盘域（%s）最大RAID成员盘数异常
            self.errMsg += common.getMsg(self.lang, "exp.tire.storage.pool.meta.tier.fail",self.domainId)
            return None
        
        isHasMetaRaid = False
        metaList = cliUtil.getVerticalCliRet(cliRet)
        for meta in metaList:
            level = meta.get("Level")
            if level == "high":
                isHasMetaRaid = True
                break
        
        if not isHasMetaRaid:
            return 0
        
        name = config.DISK_TYPE_META_LEVEL_KEY.get(diskType)
        cmd = "spa dumpobjinfo -t 4 -p %s -o %s"%(self.domainId, name)
        result, cliRet, __ = cliUtil.excuteCmdInDebugModel(self.cli, cmd, self.lang)
        self.allCliRet += cliRet

        if "no dump" in cliRet:
            return 0

        for line in cliRet.splitlines():
            if "memberDiskNum" not in line:
                continue
            
            datas = line.split(":")
            if len(datas) < 2:
                continue
            
            return config.DISK_NUM_OF_META_SPACE.get(datas[-1].strip(), None)
        
        self.setResult(cliUtil.RESULT_NOCHECK)
        #获取硬盘域（%s）最大RAID成员盘数异常
        self.errMsg += common.getMsg(self.lang, "exp.tire.storage.pool.meta.tier.fail",self.domainId)
        return None
        
    
    def getMaxRaidDiskNum(self, diskType):
        '''
        @summary: 获取最大RAID成员盘数
        '''
        metaNum = self.getMetaRaidDiskNum(diskType)
        self.logger.logInfo("Get meta space memberDiskNum:%s" %metaNum)
        if metaNum is None:
            return None
        
        maxNum = 0
        name = config.DISK_TYPE_LEVEL_KEY.get(diskType)
        for tire in self.tierInfo:
            if name == tire.get("Name"):
                num = int(tire.get("RAID Disk Number"))
                maxNum = num if num > maxNum else maxNum
        
        maxNum += self.getHotDiskNum(maxNum)
        return maxNum if maxNum > metaNum else metaNum

    def getDataSpaceRaidNum(self, diskType):
        maxNum = 0
        name = config.DISK_TYPE_LEVEL_KEY.get(diskType)
        for tire in self.tierInfo:
            if name == tire.get("Name"):
                num = int(tire.get("RAID Disk Number"))
                maxNum = num if num > maxNum else maxNum
        return maxNum


    def getNewTierSSDNum(self, expDisk):
        '''
        @summary: 获取新增层级的最小盘数{}
        @param expDisk:(eng,diskType,diskNum) 
        '''
        #F系列不检查
        if self.isFDev():
            self.logger.logInfo("The current device is F device:%s" %self.productModel)
            return 0
        
        expEng = expDisk[0]
        if expEng in self.ssdEngList:
            return 0
        
        existHdd = False
        if expEng in self.hddEngList:
            existHdd = True
        
        if self.productModel in EXPAND_TIER_CHECK_PDT_MODEL_HIGH:
            if existHdd:
                return SSD_MIN_HAVE_HDD_4_HIGH
            else:
                return SSD_MIN_NO_HDD_4_HIGH
        else:
            if existHdd:
                return SSD_MIN_HAVE_HDD_4_MID
            else:
                return SSD_MIN_NO_HDD_4_MID
            
        return 0

    def checkOneExpDisk(self,expDisk):
            '''
            @summary: 检查是否满足起扩盘数 
            @param expDisk:(eng,diskType,diskNum) 
            '''
            #最大RAID成员盘数
            expEngId = expDisk[0]
            expDiskType = expDisk[1]
            if not self.existDiskType(expEngId, expDiskType):
                self.logger.logInfo("the disk type does not exist on disk domain or exits on new engine")
                return
            
            expDiskNum = expDisk[2]
            maxRaidNum = self.getMaxRaidDiskNum(expDiskType)
            self.logger.logInfo("Get raid disk number:%s" %maxRaidNum)
            if maxRaidNum is None:
                self.logger.logInfo("The %s disk of the domain(%s) on the eng(%s) get max raid num error."%(expDiskType, self.domainId,expDiskType))
                return
            
            if expDiskNum < maxRaidNum:
                self.result = False
                dispEng = common.getDisplayEngId(expEngId, self.productModel)
                #在硬盘域%s中，引擎%s上新扩容的硬盘（类型：%）数量（数量：%s）不满足最小起扩盘数（数量：%s）。
                self.errMsg += common.getMsg(self.lang, "exp.tire.exp.disk.less",(self.domainId, dispEng, expDiskType, expDiskNum, maxRaidNum))
                return
            return
    
    def setResult(self,flag):
        if flag != True:
            self.result = flag if flag and self.result else False
    
    def getResult(self):
        return self.result, self.errMsg, self.allCliRet

    def isNlSasOrSata(self,diskType):
        if diskType in [config.DiskType.NearLine_SAS, config.DiskType.NearLine_SAS_SED, config.DiskType.SATA, config.DiskType.SATA_SED]:
            return True
        return  False

    def isSas(self,diskType):
        if diskType in [config.DiskType.SAS, config.DiskType.SAS_SED]:
            return True
        return  False

    def getExpNlSasOrSata(self):
        nlSas = []
        for exp in self.expHDDList:
            if self.isNlSasOrSata(exp[1]):
                nlSas.append(exp)
        return nlSas

    def isCheckNlSas(self, engid):
        flag = False
        for disk in self.diskList:
            dEng = common.getEngineIdByDiskId(disk.get("ID"), self.productModel)
            if str(engid) != str(dEng):
                continue
            
            ty = disk.get("Type")
            if self.isSas(ty):
                flag = False
                break

            if self.isNlSasOrSata(ty):
                flag = True
        return flag


    def checkNlSas(self):
        if self.domainId == "none":
            return

        disks = self.getExpNlSasOrSata()
        self.logger.logInfo("get NearLine SAS and SATA result:%s" % disks)
        if not disks:
            return

        pools = self.getStoragePools()
        self.logger.logInfo("get pools result:%s" % pools)
        if not pools:
            return

        # 获取存储池tier信息
        self.tierInfo = self.getTierInfo(pools)
        if self.tierInfo == None:
            return

        for exp in disks:
            engId = exp[0]
            
            checkFlag = self.isCheckNlSas(engId)
            if not checkFlag:
                self.logger.logInfo("Donot have Nearline SAS or SATA in disk domain(ID:%s) of the engine(ID:%s)."%(self.domainId,engId))
                continue
            
            expDiskType = exp[1]
            if not self.isNlSasOrSata(expDiskType):
                continue

            raidNum = self.getDataSpaceRaidNum(expDiskType)
            self.logger.logInfo("get raid num:%s"%raidNum)
            if not raidNum:
                continue

            maxNum = raidNum + self.getHotDiskNum(raidNum)
            if maxNum > exp[2]:
                self.result = False
                displayId = common.getDisplayEngId(engId,self.productModel)
                self.errMsg += common.getMsg(self.lang, "exp.tire.exp.disk.less", (self.domainId, displayId, exp[1],exp[2], maxNum))
                continue


