# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2012-. All rights reserved.

import codecs
import re


class BiosUartPrintParser(object):
    def __init__(self, file_name):
        self.file_name = file_name

    def execute(self):
        """
        解析日志文件，提取channel信息
        概念说明：1个控制器下的内存条分多个socket，一个socket有多个channel group，一个channel group 下面有多个channel
        :return:
        """
        all_socket_tables = self.pick_all_socket_tables()
        all_channel_groups = []
        for one_socket_table in all_socket_tables:
            one_socket_channel_groups = self._split_socket_to_channel_group(one_socket_table)
            all_channel_groups.extend(self._parse_one_socket_channel_group(one_socket_channel_groups))

        all_channels = []
        for channel_group in all_channel_groups:
            for channel in channel_group.values():
                # 按格式匹配厂商和批次
                match_obj = re.search(r"DIMM: (\w+)[\s\S]*ww(\d+)\s+(\d+)\s*([\w-]+)\s*", channel)
                if not match_obj:
                    match_obj = re.match(r"\s*(\w+)[\s\S]*ww(\d{2})(\d{4})\s*([\w-]+)\s*", channel)
                if match_obj:
                    all_channels.append(dict(vendor=match_obj.group(1),
                                             week=match_obj.group(2),
                                             year=match_obj.group(3),
                                             model=match_obj.group(4)))
        return all_channels

    def pick_all_socket_tables(self):
        all_socket_tables = self.pick_all_socket_tables_for_x86()
        if not all_socket_tables:
            all_socket_tables = self.pick_all_socket_tables_for_arm()
        return all_socket_tables

    def pick_all_socket_tables_for_x86(self):
        """
        针对X86设备，提取日志中内存条信息，按socket的维度进行拆分
        一个日志文件中可能有多条打印，要求取最近一条
        :return: [socket1 origin info, socket2 origin info]
        数据格式示例：
        START_DIMMINFO_TABLE
        ================================================================================================================================
        START_SOCKET_0_TABLE
        SKX  M0 - Socket 4S
        ================================================================================================================================
        S|     Channel 0      |     Channel 1      |     Channel 2      |     Channel 3      |     Channel 4      |     Channel 5      |
        ================================================================================================================================
        0|   DIMM: Micron     |   DIMM: Micron     |   Not installed    |   DIMM: Micron     |   DIMM: Micron     |   Not installed    |
         |   DRAM: Micron     |   DRAM: Micron     |                    |   DRAM: Micron     |   DRAM: Micron     |                    |
         |    RCD: IDT        |    RCD: IDT        |                    |    RCD: IDT        |    RCD: IDT        |                    |
         |  32GB(8Gbx4    DR) |  32GB(8Gbx4    DR) |                    |  32GB(8Gbx4    DR) |  32GB(8Gbx4    DR) |                    |
         | DDR4 RDIMM  R/C-B  | DDR4 RDIMM  R/C-B  |                    | DDR4 RDIMM  R/C-B  | DDR4 RDIMM  R/C-B  |                    |
         |   2666 17-17-17    |   2666 17-17-17    |                    |   2666 17-17-17    |   2666 17-17-17    |                    |
         |     ww11 2018      |     ww11 2018      |                    |     ww11 2018      |     ww11 2018      |                    |
         |36ASF4G72PZ-2G6D1   |36ASF4G72PZ-2G6D1   |                    |36ASF4G72PZ-2G6D1   |36ASF4G72PZ-2G6D1   |                    |
         |0x0000000000000000  |0x0000000000000000  |                    |0x0000000000000000  |0x0000000000000000  |                    |
         |                    |                    |                    |                    |                    |                    |
        --------------------------------------------------------------------------------------------------------------------------------
        1|   Not installed    |   Not installed    |   Not installed    |   Not installed    |   Not installed    |   Not installed    |
        --------------------------------------------------------------------------------------------------------------------------------
        STOP_SOCKET_0_TABLE
        ================================================================================================================================
        STOP_DIMMINFO_TABLE
        """
        all_socket_data = []
        temp_socket = []
        start_flag = False
        with codecs.open(self.file_name, "r") as f_obj:
            for line in f_obj:
                if "START_DIMMINFO_TABLE" in line:
                    start_flag = True
                    all_socket_data = []
                    temp_socket = []

                if "STOP_DIMMINFO_TABLE" in line:
                    start_flag = False

                if not start_flag:
                    continue

                temp_socket.append(line)
                if re.search(r"START_SOCKET_\d+_TABLE", line):
                    temp_socket = []

                if re.search(r"STOP_SOCKET_\d+_TABLE", line):
                    all_socket_data.append(temp_socket)
                    temp_socket = []
        return all_socket_data

    def pick_all_socket_tables_for_arm(self):
        """
        针对arm设备，提取日志中内存条信息，按socket的维度进行拆分
        一个日志文件中可能有多条打印，要求取最近一条

        :return: [socket1 origin info, socket2 origin info]
        DisplayDimmInfo..........................................Start
        ===========================================================================================================================================================================
        | Socekt 0                                                                                                                                                                |
        ===========================================================================================================================================================================
        | Channel |         0         |         1         |         2         |         3         |         4         |         5         |         6         |         7         |
        ===========================================================================================================================================================================
        |    0    |   DIMM: Samsung   |   DIMM: Samsung   |      NO DIMM      |      NO DIMM      |   DIMM: Samsung   |      NO DIMM      |      NO DIMM      |      NO DIMM      |
        |         |   DRAM: Samsung   |   DRAM: Samsung   |                   |                   |   DRAM: Samsung   |                   |                   |                   |
        |         |    RCD: Montage   |    RCD: Montage   |                   |                   |    RCD: Montage   |                   |                   |                   |
        |         |    DDR4 RDIMM     |    DDR4 RDIMM     |                   |                   |    DDR4 RDIMM     |                   |                   |                   |
        |         |   16GB(2RX8 1H)   |   16GB(2RX8 1H)   |                   |                   |   16GB(2RX8 1H)   |                   |                   |                   |
        |         |        2666       |        2666       |                   |                   |        2666       |                   |                   |                   |
        |         |    ww11 2019      |    ww11 2019      |                   |                   |    ww11 2019      |                   |                   |                   |
        |         | M393A2K43BB1-CTD  | M393A2K43BB1-CTD  |                   |                   | M393A2K43BB1-CTD  |                   |                   |                   |
        |         |     25440DB6      |     25440DD1      |                   |                   |     25440E14      |                   |                   |                   |
        |         |                   |                   |                   |                   |                   |                   |                   |                   |
        | Physical|      DIMM 060     |      DIMM 040     |      DIMM 020     |      DIMM 000     |      DIMM 050     |      DIMM 070     |      DIMM 010     |      DIMM 030     |
        ===========================================================================================================================================================================
        DisplayDimmInfo..........................................Done
        """
        all_socket_data = []
        temp_socket = []
        start_flag = False
        with codecs.open(self.file_name, "r") as f_obj:
            for line in f_obj:
                if re.match(r"DisplayDimmInfo.+Start", line):
                    start_flag = True
                    all_socket_data = []
                    temp_socket = []

                if re.match(r"DisplayDimmInfo.+Done", line):
                    all_socket_data.append(temp_socket)
                    start_flag = False

                if not start_flag:
                    continue

                temp_socket.append(line)
                if re.search(r"Socekt \d+", line) and temp_socket:
                    all_socket_data.append(temp_socket)
                    temp_socket = []

        return all_socket_data[1:]

    @staticmethod
    def _split_socket_to_channel_group(one_socket_info):
        """
        将socket拆分成title和channel group
        一个socket由1个title部分+多个channel group部分组成

        :param one_socket_info:
        :return: channel_groups: 第0个元素为表头，剩下的元素为数据
        """
        start_flag = False
        channel_groups = []
        temp_data = []
        for line in one_socket_info:
            line = line.strip()
            if not start_flag and re.match("^={10,}$", line):
                start_flag = True
                continue
            if not start_flag:
                continue

            if re.match("^-{10,}$", line) or re.match("^={10,}$", line):
                channel_groups.append(temp_data)
                temp_data = []
                continue

            if start_flag:
                temp_data.append(line)

        return channel_groups

    def _parse_one_socket_channel_group(self, channel_groups):
        """
        对一个socket下的channel groups原始信息解析成channel格式数据
        :param channel_groups: 一个socket下的channel groups信息
        :return: one_socket_channel_groups
        """
        head_line = channel_groups[0][0]
        heads = head_line.split("|")
        one_socket_channel_groups = []
        for data in channel_groups[1:]:
            one_group_channels = self._parse_all_channel_in_one_group(data)
            one_group_channel_dict = self._merge_head_and_channels(heads, one_group_channels)
            one_socket_channel_groups.append(one_group_channel_dict)
        return one_socket_channel_groups

    @staticmethod
    def _parse_all_channel_in_one_group(one_group_data):
        split_datas = []
        for line in one_group_data:
            split_datas.append(line.split("|"))
        # 按列倒置
        channels = []
        for col_index in range(len(split_datas[0])):
            channel_data = []
            for row_index in range(len(split_datas)):
                channel_data.append(split_datas[row_index][col_index])
            channels.append("\n".join(channel_data))
        return channels

    @staticmethod
    def _merge_head_and_channels(heads, contents):
        """
        将channel的头部和内容按列拼接起来
        :param heads:
        :param contents:
        :return:
        """
        ret = {}
        if len(heads) != len(contents):
            return {}
        for i in range(len(heads)):
            if heads[i]:
                ret[heads[i]] = contents[i]
        return ret
