# -*- coding: UTF-8 -*-
import ConfigParser
import codecs

from memory_inspect.utils import constants


class ConfigParserHandler(object):
    """文件配置"""
    ROW = "row_error"
    COLUMN = "column_error"
    BANK = "bank_error"
    section = "default"
    PREFIX_DICT = {
        ROW: "ROW_",
        COLUMN: "COLUMN_",
        BANK: "BANK_",
    }

    def __init__(self, config_file_path, logger):
        self.config_file_path = config_file_path
        self.logger = logger
        self.config = ConfigParser.SafeConfigParser(allow_no_value=True)
        self.config.read(self.config_file_path)
        self.logger.info("config items:{}".format(self.config.sections()))
        self.logged_err_info_set = set()

    def get_all_items(self):
        sections = self.config.sections()
        import collections
        all_items = collections.OrderedDict()
        for section in sections:
            all_params = self.config.items(section)
            item_dict = {}
            for one_param in all_params:
                if len(one_param) == 2:
                    item_dict.setdefault(one_param[0], one_param[1])
            all_items.setdefault(section, item_dict)
        return all_items

    def _save_config_file(self):
        with codecs.open(self.config_file_path, 'wb', encoding="utf-8") as configfile:
            self.config.write(configfile)
        self.logger.info("save config file:{}".format(self.config_file_path))

    def get_value(self, section, key, val_typ_func):
        try:
            value = val_typ_func(self.config.get(section, key))
        except Exception as e:
            err_info = "value get error!section:{}, key:{}, val_typ_func:{}, err:{}".format(
                section, key, val_typ_func, e)
            if err_info not in self.logged_err_info_set:
                # 防止重复打印日志
                self.logged_err_info_set.add(err_info)
                self.logger.error(err_info)
            value = getattr(constants, self.PREFIX_DICT.get(section, "") + key)
        return value

    @property
    def max_interval_secs(self):
        return self.get_value(self.section, "MAX_INTERVAL_SECS", val_typ_func=int)

    def min_continue_count(self, err_section):
        return self.get_value(err_section, "MIN_CONTINUE_COUNT", val_typ_func=int)

    def min_dq_error_count(self, err_section):
        return self.get_value(err_section, "MIN_DQ_ERROR_COUNT", val_typ_func=int)

    def min_err_addr_count(self, err_section):
        return self.get_value(err_section, "MIN_ERR_ADDR_COUNT", val_typ_func=int)

    @property
    def is_debug(self):
        return self.get_value(self.section, "IS_DEBUG", val_typ_func=int)
