# -*- coding: UTF-8 -*-
import json

import cliUtil
import common
import datetime, time

from cbb.frame.base import config
from cbb.frame.context import contextUtil
from cbb.frame.rest import restUtil
from cbb.frame.rest import restData

from utils import Products

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env

# 需要使用rest接口的产品版本
REST_API_MODEL_AND_VERSION_DICT = {
    tuple(config.OCEAN_PROTECT): "1.6.RC1",
    tuple(config.OCEAN_STOR_MICRO_DEVS): "1.3.RC1",
    tuple(config.DORADO_DEVS_V6): "6.1.8RC1"
}


def execute(cli):
    """
    Create date        : 2016/10/13
    Function name      : execute(cliConnection)
    Function describe  : execute repeated recovered Critical or Major alarm.
    Input              : cli--Telnet access proxy
    Return             : the checkitem is pass;cli source info
    """
    allCliRet = ""
    alarmCliRet = ""
    errMsg = ""
    # 记录检查开始时间
    startTime = time.time()

    try:
        # 检查用户是否为只读用户（只读用户限制检查）
        loginUserName = py_java_env.get("devInfoMap").get("userName")
        LOGGER.logInfo("loginUserName is %s" % str(loginUserName))
        # 刷新进度条
        common.refreshProcess(PY_JAVA_ENV, 1, LOGGER)
        isSucc, cliRet, errMsg = checkLoginGuest(cli, loginUserName, LANG)
        if isSucc != True:
            return (isSucc, cliRet, errMsg)
        # 刷新进度条
        common.refreshProcess(PY_JAVA_ENV, 2, LOGGER)
        isSucc, cliRet, errMsg, sys_time = cliUtil.getCurrentTime(cli, LANG, LOGGER)
        allCliRet += cliRet
        if not isSucc:
            errMsg = common.getMsg(LANG, "query.system.date.abnormal")
            LOGGER.logNoPass("get system date abnormal.")
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)  # 修改备注：获取设备当前时间失败
        sysDate = sys_time.split("/")[0]
        year, month, day = map(int, sysDate.split('-'))
        lastMonth = datetime.date(year, month, day) - datetime.timedelta(30)

        # 刷新进度条
        common.refreshProcess(PY_JAVA_ENV, 3, LOGGER)

        # Dorado/新融合 6.1.8及以后，二级存储1.6.0及以后，微存储1.3.0及以后，使用rest接口查询告警信息
        if is_version_for_rest():
            return check_alarm_for_rest(allCliRet, sys_time)

        # 获取最近30天内的紧急告警信息
        cmd = "show event level=critical from_time=%s/00:00:00" % lastMonth.strftime('%Y-%m-%d')
        isSucc, criticalAlarmCliRet, errMsg = cliUtil.excuteCmdTimeOutInCliMode(cli, cmd, True, LANG, 5 * 60)
        allCliRet += criticalAlarmCliRet

        # 因数据量大，5min（用户可接受的时间）内未完成查询时不再做检查
        if "TOOLKIT_SEND_CMD_TIME_OUT" in criticalAlarmCliRet or isTimeOut(startTime):
            errMsg = common.getMsg(LANG, "get.event.timeout")
            return ("NOCHECK", allCliRet, errMsg)
        # 正常告警回显Name中可能包含is not supported，造成公共方法误报不涉及
        # 加入"----"关键字识别，若包含该分隔符，则认为正常回显，消除不支持虚拟机误报
        if isSucc != True and "----" not in criticalAlarmCliRet:
            LOGGER.logSysAbnormal()
            if not isSucc:
                return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)  # 修改备注：获取最近30天内的紧急告警信息命令执行失败
            return (isSucc, allCliRet, errMsg)
        errMsg = ""
        # 刷新进度条
        common.refreshProcess(PY_JAVA_ENV, 4, LOGGER)

        # 获取最近30天内的重要告警信息
        cmd = "show event level=major from_time=%s/00:00:00" % lastMonth.strftime('%Y-%m-%d')
        isSucc, majorAlarmCliRet, errMsg = cliUtil.excuteCmdTimeOutInCliMode(cli, cmd, True, LANG, 5 * 60)
        allCliRet += majorAlarmCliRet

        # 因数据量大，5min（用户可接受的时间）内未完成查询时不再做检查
        if "TOOLKIT_SEND_CMD_TIME_OUT" in majorAlarmCliRet or isTimeOut(startTime):
            errMsg = common.getMsg(LANG, "get.event.timeout")
            return ("NOCHECK", allCliRet, errMsg)
        # 正常告警回显Name中可能包含is not supported，造成公共方法误报不涉及
        # 加入"----"关键字识别，若包含该分隔符，则认为正常回显，消除不支持虚拟机误报
        if isSucc != True and "----" not in majorAlarmCliRet:
            LOGGER.logSysAbnormal()
            if not isSucc:
                return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)  # 修改备注：获取最近30天内的重要告警信息命令执行失败
            return isSucc, allCliRet, errMsg
        errMsg = ""
        # 刷新进度条
        common.refreshProcess(PY_JAVA_ENV, 5, LOGGER)

        # 最近30天设备未产生重要及紧急告警
        if cliUtil.queryResultWithNoRecord(criticalAlarmCliRet) and cliUtil.queryResultWithNoRecord(majorAlarmCliRet):
            return (True, allCliRet, errMsg)

        alarmCliRet = criticalAlarmCliRet + majorAlarmCliRet

        alarmCliRetLines = alarmCliRet.splitlines()
        alarmNum = len(alarmCliRetLines) - 7
        if alarmNum >= 300:
            currentProcess = 5
            # 5分钟内最多查询800条记录
            if alarmNum <= 800:
                stepProcess = (1.0 / alarmNum) * (100 - currentProcess)
            else:
                stepProcess = (1.0 / 800) * (100 - currentProcess)
        alarmDict = {}
        for line in alarmCliRetLines:
            if isTimeOut(startTime):
                errMsg = common.getMsg(LANG, "get.event.timeout")
                return ("NOCHECK", allCliRet, errMsg)

            itemList = line.split()
            if len(itemList) < 1:
                continue

            alarmSequence = itemList[0].strip()
            if alarmSequence.isdigit():
                alarmDetail, detailCliRet = common.qryEventDetail(cli, alarmSequence)
                alarmLineList = alarmDict.get(alarmDetail, [])
                alarmLineList.append(line.strip())
                alarmDict[alarmDetail] = alarmLineList
                allCliRet += detailCliRet
                if alarmNum >= 300:
                    currentProcess += stepProcess
                    common.refreshProcess(py_java_env, int(currentProcess), LOGGER)

        for alarmDetail in alarmDict:
            alarmLineList = alarmDict.get(alarmDetail)
            alarmOccurTimes = len(alarmLineList)
            alarmLevel = alarmLineList[0].split()[1]

            if alarmLevel == "Major":
                if alarmOccurTimes == 3:
                    errMsg += common.getMsg(LANG, "alarm.3times.repeat.Occur",
                                            (alarmOccurTimes, "\n".join(alarmLineList)))
                if alarmOccurTimes > 3:
                    errMsg += common.getMsg(LANG, "alarm.than3times.repeat.Occur",
                                            (alarmOccurTimes, 3, "\n".join(alarmLineList[0:3])))

            if alarmLevel == "Critical":
                if alarmOccurTimes == 2:
                    errMsg += common.getMsg(LANG, "alarm.3times.repeat.Occur",
                                            (alarmOccurTimes, "\n".join(alarmLineList)))
                if alarmOccurTimes > 2:
                    errMsg += common.getMsg(LANG, "alarm.than3times.repeat.Occur",
                                            (alarmOccurTimes, 2, "\n".join(alarmLineList[0:2])))

        if errMsg:
            LOGGER.logNoPass("System exists repeat alarm")
            return ('WARNING', allCliRet, errMsg)
        else:
            return (True, allCliRet, errMsg)

    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))


def isTimeOut(startTime):
    '''
    @summary: 检查项是否执行超过5min
    '''
    if (time.time() - startTime) > 5 * 60:
        return True

    return False


def checkLoginGuest(cli, loginUserName, LANG):
    '''
    @summary: 检查用户是否为只读用户
    '''
    guestUserLevel = "Guest"
    cmd = "show user user_name=%s" % (loginUserName)
    isSuccess, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if isSuccess != True:
        LOGGER.logSysAbnormal()
        return isSuccess, cliRet, errMsg

    dictList = cliUtil.getHorizontalCliRet(cliRet)
    if len(dictList) == 0:
        errMsg = common.getMsg(LANG, "query.result.abnormal")
        LOGGER.logNoPass("Cannot get information about user")
        return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)  # 修改备注：检查用户是否为只读用户时，解析回显时失败

    level = dictList[0].get("Level", "")

    if level.upper() == guestUserLevel.upper():
        LOGGER.logNoPass("loginUser.name.level.guest.warning")
        return cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "loginUser.name.level.guest.warning")

    return True, cliRet, ""


def is_version_for_rest():
    """
    检查是否为需要rest接口查询的版本
    @return: 是：使用rest接口的版本， 否：不使用rest接口的版本
    """
    context = contextUtil.getContext(PY_JAVA_ENV)
    product_model = contextUtil.getDevType(context)
    product_version = contextUtil.getCurVersion(context)
    for model, version in REST_API_MODEL_AND_VERSION_DICT.items():
        if product_model in model and Products.compareVersion(product_version, version) >= 0:
            return True
    return False


def check_alarm_for_rest(all_cli_ret, sys_date):
    """
    通过rest接口检查告警
    @param all_cli_ret: 回显信息
    @param sys_date: 当前时间
    """
    all_ret = [all_cli_ret]
    flag, alarm_records = get_alarm_infos(all_ret, sys_date)
    if not flag:
        return cliUtil.RESULT_NOCHECK, "\n".join(all_ret), common.getMsg(LANG, "get.event.timeout")
    # 构建告警详情和告警信息字典
    alarm_dict = dict()
    for alarm_record in alarm_records:
        # 修复html上的乱码,最小化改动
        alarm_detail = alarm_record.get("detail").replace("&#40;", "(").replace("&#41;", ")")
        alarm_info = alarm_dict.get(alarm_detail, [])
        alarm_info.append(alarm_record)
        alarm_dict[alarm_detail] = alarm_info
    # 设置错误信息
    err_msg = ""
    for alarm_detail, alarm_info in alarm_dict.items():
        alarm_level = alarm_info[0].get("level")
        alarm_num = len(alarm_info)
        LOGGER.logInfo("alarm_num is %s" % str(alarm_num))
        # 重要告警major=5 次数>=3,紧急告警critical=6，次数>=2
        if str(alarm_level) == "5" and alarm_num >= 3:
            err_msg += common.getMsg(LANG, "major.alarm.3times.repeat.Occur", (alarm_num, alarm_detail))
        if str(alarm_level) == "6" and alarm_num >= 2:
            err_msg += common.getMsg(LANG, "critical.alarm.3times.repeat.Occur", (alarm_num, alarm_detail))
    if not err_msg:
        return True, "\n".join(all_ret), err_msg
    return cliUtil.RESULT_WARNING, "\n".join(all_ret), err_msg


def get_alarm_infos(all_ret, sys_date):
    """
    获取告警信息
    @param all_ret: 回显
    @param sys_date: 当前时间
    @return: 告警信息
    """
    major_alarm_rest_url = "alarm/historyalarm?filter=level::5%20and%20startTime:[{},{}]&range=[{}-{}]"
    critical_alarm_rest_url = "alarm/historyalarm?filter=level::6%20and%20startTime:[{},{}]&range=[{}-{}]"
    context = contextUtil.getContext(PY_JAVA_ENV)
    rest = contextUtil.getRest(context)
    alarm_records = list()
    major_alarm = get_someone_level_alarm_info(sys_date, major_alarm_rest_url, all_ret, rest)
    alarm_records.extend(major_alarm)
    critical_alarm = get_someone_level_alarm_info(sys_date, critical_alarm_rest_url, all_ret, rest)
    alarm_records.extend(critical_alarm)
    return True, alarm_records


def get_someone_level_alarm_info(sys_date, alarm_rest_url, all_ret, rest):
    """
    获取某个指定等级的告警信息
    @param sys_date: 当前时间
    @param alarm_rest_url: url地址
    @param all_ret: 回显
    @param rest: rest连接
    @return: 告警信息
    """

    end_time = get_timestamp(sys_date)
    # 开始时间为当前时间前30天
    start_time = end_time - 30 * 24 * 60 * 60
    alarm_records = list()
    index = 0
    check_start_time = time.time()
    # 每次最多查询250条，轮询查询，直至查完,五分钟没有查完，则上报未完成检查
    while True:
        if time.time() - check_start_time > 300:
            return alarm_records
        url = alarm_rest_url.format(start_time, end_time, index, index + 250)
        all_ret.append(str(url))
        uri_param_dict = restUtil.CommonRest.getUriParamDict(url)
        ret_obj = restUtil.CommonRest.execCmd(rest, uri_param_dict, {}, restData.RestCfg.RestMethod.GET)
        all_ret.append(json.dumps(ret_obj, ensure_ascii=False))
        data = ret_obj.get("data")
        if not data:
            break
        index += 250
        alarm_records.extend(data)
        if len(data) < 250:
            break
    return alarm_records


def get_timestamp(time_str):
    """将时间字符串转换为时间戳

    :param time_str: 时间格式为 2024-01-11/19:31:24
    :return:
    """
    time_array = time.strptime(time_str, "%Y-%m-%d/%H:%M:%S")
    timestamp = int(time.mktime(time_array))
    return timestamp