# -*- coding: UTF-8 -*-
import time
import re
import cliUtil
import common
from common_utils import is_super_administrator
from common_utils import get_err_msg

# 当前时间和文件更新时间的时间差
TIME_DIFFERENCE = 3600
MONTH_TABLE = {
    "Jan": "01",
    "Feb": "02",
    "Mar": "03",
    "Apr": "04",
    "May": "05",
    "Jun": "06",
    "Jul": "07",
    "Aug": "08",
    "Sept": "09",
    "Sep": "09",
    "Oct": "10",
    "Nov": "11",
    "Dec": "12"
}


def execute(cli):
    """
    BMC日志状态检查
    """
    check_bmc_syslog = CheckBMCSysLog(cli)
    flag, err_msg = check_bmc_syslog.execute_check()
    return \
        flag, "\n".join(check_bmc_syslog.cli_ret_list), err_msg


class CheckBMCSysLog:
    def __init__(self, cli):
        self.cli = cli
        self.LANG = common.getLang(py_java_env)
        self.LOGGER = common.getLogger(PY_LOGGER, __file__)
        self.PY_JAVA_ENV = py_java_env
        self.cli_ret_list = []

    def execute_check(self):
        try:
            # 不是超级管理员账号,报未完成检查
            if not is_super_administrator(
                    self.cli,
                    self.PY_JAVA_ENV.get("devInfoMap").get("userName"),
                    self.LANG):
                err_msg = common.getMsg(
                    self.LANG, "loginUser.name.level.must.be.super.admin")
                return cliUtil.RESULT_NOCHECK, "", err_msg
            # 获取当前设备系统时间
            flag, cli_ret, err_msg, product_model, sys_date = \
                cliUtil.getOEMproductModelAndCurSysDate(self.cli, self.LANG)
            self.cli_ret_list.append(cli_ret)
            self.LOGGER.logInfo("sys_date: {}".format(sys_date))
            if flag is not True:
                self.LOGGER.logNoPass("get system date abnormal.")
                return False, err_msg
            # 获取cpldreg以及secure_curlog.tar.gz文件更新时间
            cpldreg_date, secure_curlog_date = self.get_file_date()
            if not cpldreg_date and not secure_curlog_date:
                return False, get_err_msg(
                    self.LANG, "failed.to.collect.BMC.log")

            file_dates = (cpldreg_date, secure_curlog_date)
            # 文件更新时间和当前时间的时间差是否大于1小时
            if self.is_the_date_difference_greater_than_one_hour(
                    sys_date, file_dates):
                return False, get_err_msg(
                    self.LANG, "failed.to.collect.BMC.log")
            return True, ""
        except Exception as ex:
            self.LOGGER.logError(str(ex))
            err_msg = common.getMsg(self.LANG, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK, err_msg
        finally:
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.LANG)
            self.LOGGER.logInfo("enter cli mode from some model {}".format(ret))
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.LANG)

    def get_file_date(self):
        """
        获取cpldreg以及secure_curlog.tar.gz文件更新时间
        """
        cpldreg_date = ""
        secure_curlog_date = ""
        cmd = "ls -l /OSM/log/cur_debug/pangea/BMC/tmp/BMCLOG/local"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInMinisystemModel(self.cli, cmd, self.LANG)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            self.LOGGER.logNoPass("get file date abnormal.")
            return cpldreg_date, secure_curlog_date

        for line in cli_ret.splitlines():
            regx = "(?P<time>\w+\s+\d+\s+\d+:\d+)\s+(cpldreg|secure_curlog.tar.gz)"
            cpldreg_date_list = re.compile(regx).search(line)

            if cpldreg_date_list and "cpldreg" in line:
                cpldreg_date = cpldreg_date_list.groupdict().get("time")
            if cpldreg_date_list and "secure_curlog.tar.gz" in line:
                secure_curlog_date = cpldreg_date_list.groupdict().get("time")
        self.LOGGER.logInfo("cpldreg_date: {},secure_curlog_date: {}".format(
            str(cpldreg_date), str(secure_curlog_date)))
        return cpldreg_date, secure_curlog_date

    def is_the_date_difference_greater_than_one_hour(self, sys_date,
                                                     file_dates):
        """
        判断文件更新时间和当前时间的时间差是否大于1小时
        """
        # 设备时间格式：2021-01-01/00:44:02 UTC+08:00
        regx = "(\d+-\d+-\d+/\d+:\d+:\d+)"
        date_list = re.compile(regx).findall(sys_date)
        if not date_list:
            return False
        current_date = date_list[0]
        year_id = current_date[0:4]
        time_array = time.strptime(current_date, "%Y-%m-%d/%H:%M:%S")
        current_time = time.mktime(time_array)
        for file_date in file_dates:
            if not file_date:
                continue
            file_time = self.get_time(file_date, year_id)
            if not file_time:
                continue
            if int(current_time - file_time) > TIME_DIFFERENCE:
                return True
        return False

    @staticmethod
    def get_time(date, year_id):
        """
        转换为时间戳
        """
        # 文件更新时间格式：Nov 27 14:05
        current_date = "{} {}".format(year_id, date)

        for month_english, month_str in MONTH_TABLE.items():
            if month_english in current_date:
                current_date = current_date.replace(month_english, month_str)
        time_array = time.strptime(str(current_date), "%Y %m %d %H:%M")
        return time.mktime(time_array)
